/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.geospatial;

import com.esri.core.geometry.Point;
import com.esri.core.geometry.ogc.OGCGeometry;
import com.facebook.presto.geospatial.GeometryType;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.google.common.base.Joiner;
import java.util.EnumSet;
import java.util.Set;

public class SphericalGeographyUtils {
    public static final double EARTH_RADIUS_KM = 6371.01;
    public static final double EARTH_RADIUS_M = 6371010.0;
    private static final float MIN_LATITUDE = -90.0f;
    private static final float MAX_LATITUDE = 90.0f;
    private static final float MIN_LONGITUDE = -180.0f;
    private static final float MAX_LONGITUDE = 180.0f;
    private static final Joiner OR_JOINER = Joiner.on((String)" or ");
    private static final Set<GeometryType> ALLOWED_SPHERICAL_DISTANCE_TYPES = EnumSet.of(GeometryType.POINT);

    private SphericalGeographyUtils() {
    }

    public static void checkLatitude(double latitude) {
        if (Double.isNaN(latitude) || Double.isInfinite(latitude) || latitude < -90.0 || latitude > 90.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Latitude must be between -90 and 90");
        }
    }

    public static void checkLongitude(double longitude) {
        if (Double.isNaN(longitude) || Double.isInfinite(longitude) || longitude < -180.0 || longitude > 180.0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Longitude must be between -180 and 180");
        }
    }

    public static Double sphericalDistance(OGCGeometry leftGeometry, OGCGeometry rightGeometry) {
        if (leftGeometry.isEmpty() || rightGeometry.isEmpty()) {
            return null;
        }
        SphericalGeographyUtils.validateSphericalType("ST_Distance", leftGeometry, ALLOWED_SPHERICAL_DISTANCE_TYPES);
        SphericalGeographyUtils.validateSphericalType("ST_Distance", rightGeometry, ALLOWED_SPHERICAL_DISTANCE_TYPES);
        Point leftPoint = (Point)leftGeometry.getEsriGeometry();
        Point rightPoint = (Point)rightGeometry.getEsriGeometry();
        return SphericalGeographyUtils.greatCircleDistance(leftPoint.getY(), leftPoint.getX(), rightPoint.getY(), rightPoint.getX()) * 1000.0;
    }

    public static double greatCircleDistance(double latitude1, double longitude1, double latitude2, double longitude2) {
        SphericalGeographyUtils.checkLatitude(latitude1);
        SphericalGeographyUtils.checkLongitude(longitude1);
        SphericalGeographyUtils.checkLatitude(latitude2);
        SphericalGeographyUtils.checkLongitude(longitude2);
        double radianLatitude1 = Math.toRadians(latitude1);
        double radianLatitude2 = Math.toRadians(latitude2);
        double sin1 = Math.sin(radianLatitude1);
        double cos1 = Math.cos(radianLatitude1);
        double sin2 = Math.sin(radianLatitude2);
        double cos2 = Math.cos(radianLatitude2);
        double deltaLongitude = Math.toRadians(longitude1) - Math.toRadians(longitude2);
        double cosDeltaLongitude = Math.cos(deltaLongitude);
        double t1 = cos2 * Math.sin(deltaLongitude);
        double t2 = cos1 * sin2 - sin1 * cos2 * cosDeltaLongitude;
        double t3 = sin1 * sin2 + cos1 * cos2 * cosDeltaLongitude;
        return Math.atan2(Math.sqrt(t1 * t1 + t2 * t2), t3) * 6371.01;
    }

    public static void validateSphericalType(String function, OGCGeometry geometry, Set<GeometryType> validTypes) {
        GeometryType type = GeometryType.getForEsriGeometryType(geometry.geometryType());
        if (!validTypes.contains((Object)type)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("When applied to SphericalGeography inputs, %s only supports %s. Input type is: %s", new Object[]{function, OR_JOINER.join(validTypes), type}));
        }
    }
}

