/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.geospatial;

import com.facebook.presto.geospatial.GeometryUtils;
import com.google.common.base.Verify;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.airlift.slice.SliceOutput;
import java.util.ArrayList;
import java.util.Objects;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;

public class JtsGeometryUtils {
    private static final String JTS_POINT = "Point";
    private static final String JTS_POLYGON = "Polygon";
    private static final String JTS_LINESTRING = "LineString";
    private static final String JTS_MULTI_POINT = "MultiPoint";
    private static final String JTS_MULTI_POLYGON = "MultiPolygon";
    private static final String JTS_MULTI_LINESTRING = "MultiLineString";
    private static final String JTS_GEOMETRY_COLLECTION = "GeometryCollection";
    private static final GeometryFactory GEOMETRY_FACTORY = new GeometryFactory();

    private JtsGeometryUtils() {
    }

    public static Geometry deserialize(Slice shape) {
        if (shape == null) {
            return null;
        }
        BasicSliceInput input = shape.getInput();
        byte geometryType = input.readByte();
        if (geometryType != GeometryUtils.GeometryTypeName.GEOMETRY_COLLECTION.code()) {
            return JtsGeometryUtils.readGeometry((SliceInput)input);
        }
        if (input.available() == 0) {
            return GEOMETRY_FACTORY.createGeometryCollection();
        }
        ArrayList<Geometry> geometries = new ArrayList<Geometry>();
        while (input.available() > 0) {
            int length = input.readInt();
            geometries.add(JtsGeometryUtils.readGeometry((SliceInput)input.readSlice(length).getInput()));
        }
        return GEOMETRY_FACTORY.createGeometryCollection(geometries.toArray(new Geometry[0]));
    }

    public static Slice serialize(Geometry geometry) {
        DynamicSliceOutput output = new DynamicSliceOutput(100);
        boolean inGeometryCollection = geometry.getGeometryType().equals(JTS_GEOMETRY_COLLECTION);
        if (inGeometryCollection) {
            output.writeByte(GeometryUtils.GeometryTypeName.GEOMETRY_COLLECTION.code());
            for (int i = 0; i < geometry.getNumGeometries(); ++i) {
                JtsGeometryUtils.writeGeometry(geometry.getGeometryN(i), (SliceOutput)output, true);
            }
        } else {
            JtsGeometryUtils.writeGeometry(geometry, (SliceOutput)output, false);
        }
        return output.slice();
    }

    private static void writeGeometry(Geometry geometry, SliceOutput output, boolean inGeometryCollection) {
        switch (geometry.getGeometryType()) {
            case "Point": {
                JtsGeometryUtils.writePoint((Point)geometry, output, inGeometryCollection);
                break;
            }
            case "MultiPoint": {
                JtsGeometryUtils.writeMultiPoint((MultiPoint)geometry, output, inGeometryCollection);
                break;
            }
            case "LineString": 
            case "MultiLineString": {
                JtsGeometryUtils.writePolyline(geometry, output, inGeometryCollection);
                break;
            }
            case "Polygon": 
            case "MultiPolygon": {
                JtsGeometryUtils.writePolygon(geometry, output, inGeometryCollection);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported geometry type : " + geometry.getGeometryType());
            }
        }
    }

    private static Geometry readGeometry(SliceInput input) {
        Objects.requireNonNull(input, "input is null");
        int geometryType = input.readInt();
        switch (EsriShapeType.valueOf(geometryType)) {
            case POINT: {
                return JtsGeometryUtils.readPoint(input);
            }
            case POLYLINE: {
                return JtsGeometryUtils.readPolyline(input);
            }
            case POLYGON: {
                return JtsGeometryUtils.readPolygon(input);
            }
            case MULTI_POINT: {
                return JtsGeometryUtils.readMultiPoint(input);
            }
        }
        throw new UnsupportedOperationException("Invalid geometry type: " + geometryType);
    }

    private static Point readPoint(SliceInput input) {
        Objects.requireNonNull(input, "input is null");
        Coordinate coordinates = JtsGeometryUtils.readCoordinate(input);
        if (GeometryUtils.isEsriNaN(coordinates.x) || GeometryUtils.isEsriNaN(coordinates.y)) {
            return GEOMETRY_FACTORY.createPoint();
        }
        return GEOMETRY_FACTORY.createPoint(coordinates);
    }

    private static void writePoint(Point point, SliceOutput output, boolean inGeometryCollection) {
        Objects.requireNonNull(output, "output is null");
        if (inGeometryCollection) {
            output.writeInt(20);
        } else {
            output.writeByte(GeometryUtils.GeometryTypeName.POINT.code());
        }
        output.writeInt(EsriShapeType.POINT.code);
        if (!point.isEmpty()) {
            JtsGeometryUtils.writeCoordinate(point.getCoordinate(), output);
        } else {
            JtsGeometryUtils.writeCoordinate(new Coordinate(Double.NaN, Double.NaN), output);
        }
    }

    private static Coordinate readCoordinate(SliceInput input) {
        Objects.requireNonNull(input, "input is null");
        return new Coordinate(input.readDouble(), input.readDouble());
    }

    private static void writeCoordinate(Coordinate coordinate, SliceOutput output) {
        Objects.requireNonNull(output, "output is null");
        output.writeDouble(coordinate.x);
        output.writeDouble(coordinate.y);
    }

    private static void writeCoordinates(Coordinate[] coordinates, SliceOutput output) {
        Objects.requireNonNull(coordinates, "coordinates is null");
        Objects.requireNonNull(output, "output is null");
        for (Coordinate coordinate : coordinates) {
            JtsGeometryUtils.writeCoordinate(coordinate, output);
        }
    }

    private static Geometry readMultiPoint(SliceInput input) {
        Objects.requireNonNull(input, "input is null");
        JtsGeometryUtils.skipEnvelope(input);
        int pointCount = input.readInt();
        Point[] points = new Point[pointCount];
        for (int i = 0; i < pointCount; ++i) {
            points[i] = JtsGeometryUtils.readPoint(input);
        }
        return GEOMETRY_FACTORY.createMultiPoint(points);
    }

    private static void writeMultiPoint(MultiPoint geometry, SliceOutput output, boolean inGeometryCollection) {
        Objects.requireNonNull(geometry, "geometry is null");
        Objects.requireNonNull(output, "output is null");
        int numPoints = geometry.getNumPoints();
        if (inGeometryCollection) {
            output.writeInt(40 + 16 * numPoints);
        } else {
            output.writeByte(GeometryUtils.GeometryTypeName.MULTI_POINT.code());
        }
        output.writeInt(EsriShapeType.MULTI_POINT.code);
        JtsGeometryUtils.writeEnvelope(geometry.getEnvelopeInternal(), output);
        output.writeInt(numPoints);
        for (Coordinate coordinate : geometry.getCoordinates()) {
            JtsGeometryUtils.writeCoordinate(coordinate, output);
        }
    }

    private static Geometry readPolyline(SliceInput input) {
        int i;
        Objects.requireNonNull(input, "input is null");
        JtsGeometryUtils.skipEnvelope(input);
        int partCount = input.readInt();
        if (partCount == 0) {
            return GEOMETRY_FACTORY.createLineString();
        }
        int pointCount = input.readInt();
        LineString[] lineStrings = new LineString[partCount];
        int[] partLengths = new int[partCount];
        input.readInt();
        if (partCount > 1) {
            partLengths[0] = input.readInt();
            for (i = 1; i < partCount - 1; ++i) {
                partLengths[i] = input.readInt() - partLengths[i - 1];
            }
            partLengths[partCount - 1] = pointCount - partLengths[partCount - 2];
        } else {
            partLengths[0] = pointCount;
        }
        for (i = 0; i < partCount; ++i) {
            lineStrings[i] = GEOMETRY_FACTORY.createLineString(JtsGeometryUtils.readCoordinates(input, partLengths[i]));
        }
        if (lineStrings.length == 1) {
            return lineStrings[0];
        }
        return GEOMETRY_FACTORY.createMultiLineString(lineStrings);
    }

    private static void writePolyline(Geometry geometry, SliceOutput output, boolean inGeometryCollection) {
        int numParts;
        Objects.requireNonNull(geometry, "polyline is null");
        Objects.requireNonNull(output, "output is null");
        int numPoints = geometry.getNumPoints();
        switch (geometry.getGeometryType()) {
            case "MultiLineString": {
                numParts = geometry.getNumGeometries();
                break;
            }
            case "LineString": {
                numParts = numPoints > 0 ? 1 : 0;
                break;
            }
            default: {
                throw new IllegalArgumentException("Expected LineString or MultiLineString, but got " + geometry.getGeometryType());
            }
        }
        if (inGeometryCollection) {
            output.writeInt(44 + 4 * numParts + 16 * numPoints);
        } else {
            switch (geometry.getGeometryType()) {
                case "LineString": {
                    output.writeByte(GeometryUtils.GeometryTypeName.LINE_STRING.code());
                    break;
                }
                case "MultiLineString": {
                    output.writeByte(GeometryUtils.GeometryTypeName.MULTI_LINE_STRING.code());
                }
            }
        }
        output.writeInt(EsriShapeType.POLYLINE.code);
        JtsGeometryUtils.writeEnvelope(geometry.getEnvelopeInternal(), output);
        output.writeInt(numParts);
        output.writeInt(numPoints);
        if (numParts > 0) {
            output.writeInt(0);
            for (int i = 0; i < numParts - 1; ++i) {
                output.writeInt(geometry.getGeometryN(i).getNumPoints());
            }
        }
        JtsGeometryUtils.writeCoordinates(geometry.getCoordinates(), output);
    }

    private static Coordinate[] readCoordinates(SliceInput input, int count) {
        Objects.requireNonNull(input, "input is null");
        Verify.verify((count > 0 ? 1 : 0) != 0);
        Coordinate[] coordinates = new Coordinate[count];
        for (int i = 0; i < count; ++i) {
            coordinates[i] = JtsGeometryUtils.readCoordinate(input);
        }
        return coordinates;
    }

    private static Geometry readPolygon(SliceInput input) {
        Objects.requireNonNull(input, "input is null");
        JtsGeometryUtils.skipEnvelope(input);
        int partCount = input.readInt();
        if (partCount == 0) {
            return GEOMETRY_FACTORY.createPolygon();
        }
        int pointCount = input.readInt();
        int[] startIndexes = new int[partCount];
        for (int i = 0; i < partCount; ++i) {
            startIndexes[i] = input.readInt();
        }
        int[] partLengths = new int[partCount];
        if (partCount > 1) {
            partLengths[0] = startIndexes[1];
            for (int i = 1; i < partCount - 1; ++i) {
                partLengths[i] = startIndexes[i + 1] - startIndexes[i];
            }
        }
        partLengths[partCount - 1] = pointCount - startIndexes[partCount - 1];
        LinearRing shell = null;
        ArrayList<LinearRing> holes = new ArrayList<LinearRing>();
        ArrayList<Polygon> polygons = new ArrayList<Polygon>();
        for (int i = 0; i < partCount; ++i) {
            Coordinate[] coordinates = JtsGeometryUtils.readCoordinates(input, partLengths[i]);
            if (!JtsGeometryUtils.isClockwise(coordinates)) {
                holes.add(GEOMETRY_FACTORY.createLinearRing(coordinates));
                continue;
            }
            if (shell != null) {
                polygons.add(GEOMETRY_FACTORY.createPolygon(shell, holes.toArray(new LinearRing[0])));
            }
            shell = GEOMETRY_FACTORY.createLinearRing(coordinates);
            holes.clear();
        }
        polygons.add(GEOMETRY_FACTORY.createPolygon(shell, holes.toArray(new LinearRing[0])));
        if (polygons.size() == 1) {
            return (Geometry)polygons.get(0);
        }
        return GEOMETRY_FACTORY.createMultiPolygon(polygons.toArray(new Polygon[0]));
    }

    private static void writePolygon(Geometry geometry, SliceOutput output, boolean inGeometryCollection) {
        Objects.requireNonNull(geometry, "geometry is null");
        Objects.requireNonNull(output, "output is null");
        if (!geometry.getGeometryType().equals(JTS_POLYGON) && !geometry.getGeometryType().equals(JTS_MULTI_POLYGON)) {
            throw new IllegalArgumentException("Expected LineString or MultiLineString, but got " + geometry.getGeometryType());
        }
        int numGeometries = geometry.getNumGeometries();
        int numParts = 0;
        int numPoints = geometry.getNumPoints();
        for (int i = 0; i < numGeometries; ++i) {
            Polygon polygon = (Polygon)geometry.getGeometryN(i);
            if (polygon.getNumPoints() <= 0) continue;
            numParts += polygon.getNumInteriorRing() + 1;
        }
        if (inGeometryCollection) {
            output.writeInt(36 + 4 * (2 + numParts) + 16 * numPoints);
        } else {
            switch (geometry.getGeometryType()) {
                case "MultiPolygon": {
                    output.writeByte(GeometryUtils.GeometryTypeName.MULTI_POLYGON.code());
                    break;
                }
                case "Polygon": {
                    output.writeByte(GeometryUtils.GeometryTypeName.POLYGON.code());
                }
            }
        }
        output.writeInt(EsriShapeType.POLYGON.code);
        JtsGeometryUtils.writeEnvelope(geometry.getEnvelopeInternal(), output);
        output.writeInt(numParts);
        output.writeInt(numPoints);
        if (numParts > 0) {
            int partIndex = 0;
            output.writeInt(partIndex);
            for (int i = 0; i < numGeometries; ++i) {
                Polygon polygon = (Polygon)geometry.getGeometryN(i);
                if (polygon.getNumPoints() <= 0) continue;
                int numInteriorRing = polygon.getNumInteriorRing();
                if (i == numGeometries - 1 && numInteriorRing == 0) break;
                output.writeInt(partIndex += polygon.getExteriorRing().getNumPoints());
                for (int j = 0; j < numInteriorRing && (i != numGeometries - 1 || j != numInteriorRing - 1); ++j) {
                    output.writeInt(partIndex += polygon.getInteriorRingN(j).getNumPoints());
                }
            }
        }
        JtsGeometryUtils.writeCoordinates(geometry.getCoordinates(), output);
    }

    private static void skipEnvelope(SliceInput input) {
        Objects.requireNonNull(input, "input is null");
        input.skip(32L);
    }

    private static void writeEnvelope(Envelope envelope, SliceOutput output) {
        Objects.requireNonNull(envelope, "envelope is null");
        Objects.requireNonNull(output, "output is null");
        output.writeDouble(envelope.getMinX());
        output.writeDouble(envelope.getMinY());
        output.writeDouble(envelope.getMaxX());
        output.writeDouble(envelope.getMaxY());
    }

    private static boolean isClockwise(Coordinate[] coordinates) {
        double area = 0.0;
        for (int i = 1; i < coordinates.length; ++i) {
            area += (coordinates[i].x - coordinates[i - 1].x) * (coordinates[i].y + coordinates[i - 1].y);
        }
        return (area += (coordinates[0].x - coordinates[coordinates.length - 1].x) * (coordinates[0].y + coordinates[coordinates.length - 1].y)) > 0.0;
    }

    private static enum EsriShapeType {
        POINT(1),
        POLYLINE(3),
        POLYGON(5),
        MULTI_POINT(8);

        final int code;

        private EsriShapeType(int code) {
            this.code = code;
        }

        static EsriShapeType valueOf(int code) {
            for (EsriShapeType type : EsriShapeType.values()) {
                if (type.code != code) continue;
                return type;
            }
            throw new IllegalArgumentException("Unsupported ESRI shape code: " + code);
        }
    }
}

