/*
 * Copyright 2018 The Exonum Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.exonum.binding.util;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * A loader of the native shared library with Exonum framework bindings.
 *
 * <p>To have library java_bindings available by its name,
 * add a path to the folder containing it to <code>java.library.path</code> property,
 * e.g.: <code>java -Djava.library.path=rust/target/release …</code>
 */
public final class LibraryLoader {

  private static final String BINDING_LIB_NAME = "java_bindings";
  private static final String JAVA_LIBRARY_PATH_PROPERTY = "java.library.path";
  private static final String DYNAMIC_LIBRARIES_ENV_VAR_WINDOWS = "PATH";
  private static final String DYNAMIC_LIBRARIES_ENV_VAR_UNIX = "LD_LIBRARY_PATH";

  private static final Logger logger = LogManager.getLogger(LibraryLoader.class);

  /**
   * Loads the native library with Exonum framework bindings.
   */
  public static void load() {
    loadOnce();
  }

  private static void loadOnce() {
    try {
      System.loadLibrary(BINDING_LIB_NAME);
    } catch (UnsatisfiedLinkError e) {
      logger.error("Failed to load '{}' library: {}…\n{}",
          BINDING_LIB_NAME, e, extraLibLoadErrorInfo());
      throw e;
    }
  }

  private static String extraLibLoadErrorInfo() {
    String javaLibPath = System.getProperty(JAVA_LIBRARY_PATH_PROPERTY);
    String dynamicLibVar = dynamicLibrariesEnvVar();
    String dynamicLibPath = System.getenv(dynamicLibVar);
    // todo: clarify this message when LD_LIBRARY_PATH becomes required.
    return "java.library.path=" + javaLibPath + ", \n"
        + dynamicLibVar + "=" + dynamicLibPath
        + "\nMake sure that:\n"
        + "1. The path to a directory containing '" + BINDING_LIB_NAME
        + "' dynamic library image is included in either java.library.path system property or "
        + dynamicLibVar + " environment variable.\n"
        + "2. The paths to directories containing dynamic libraries required by '"
        + BINDING_LIB_NAME + "', if any, are included in " + dynamicLibVar
        + " environment variable";
  }

  private static String dynamicLibrariesEnvVar() {
    if (OsInfo.isWindows()) {
      return DYNAMIC_LIBRARIES_ENV_VAR_WINDOWS;
    } else {
      return DYNAMIC_LIBRARIES_ENV_VAR_UNIX;
    }
  }

  private LibraryLoader() {}
}
