/*
 * Copyright 2019 The Exonum Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.exonum.binding.core.storage.indices;

import static com.exonum.binding.core.storage.indices.StoragePreconditions.checkElementIndex;
import static com.exonum.binding.core.storage.indices.StoragePreconditions.checkNoNulls;
import static com.google.common.base.Preconditions.checkArgument;

import com.exonum.binding.common.serialization.CheckingSerializerDecorator;
import com.exonum.binding.core.proxy.NativeHandle;
import com.exonum.binding.core.storage.database.AbstractAccess;
import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * An abstract class for list indices implementing {@link ListIndex} interface.
 *
 * <p>Implements all methods from ListIndex.
 */
abstract class AbstractListIndexProxy<T> extends AbstractIndexProxy implements ListIndex<T> {

  final CheckingSerializerDecorator<T> serializer;

  AbstractListIndexProxy(NativeHandle nativeHandle, IndexAddress address, AbstractAccess access,
                         CheckingSerializerDecorator<T> userSerializer) {
    super(nativeHandle, address, access);
    this.serializer = userSerializer;
  }

  @Override
  public final void add(T e) {
    notifyModified();
    byte[] dbElement = serializer.toBytes(e);
    nativeAdd(getNativeHandle(), dbElement);
  }

  @Override
  public void addAll(Collection<? extends T> elements) {
    notifyModified();
    checkNoNulls(elements);
    addAllUnchecked(elements);
  }

  private void addAllUnchecked(Collection<? extends T> elements) {
    // Cache the nativeHandle to avoid repeated 'isValid' checks.
    // It's OK to do that during this call, as this class is not thread-safe.
    long nativeHandle = getNativeHandle();
    elements.stream()
        .map(serializer::toBytes)
        .forEach((e) -> nativeAdd(nativeHandle, e));
  }

  @Override
  public final void set(long index, T e) {
    checkElementIndex(index, size());
    notifyModified();
    byte[] dbElement = serializer.toBytes(e);
    nativeSet(getNativeHandle(), index, dbElement);
  }

  @Override
  public final T get(long index) {
    checkElementIndex(index, size());
    byte[] e = nativeGet(getNativeHandle(), index);
    return serializer.fromBytes(e);
  }

  @Override
  public final T getLast() {
    byte[] e = nativeGetLast(getNativeHandle());
    // This method does not check if the list is empty first to use only a single native call.
    if (e == null) {
      throw new NoSuchElementException("List is empty");
    }
    return serializer.fromBytes(e);
  }

  @Override
  public T removeLast() {
    notifyModified();
    byte[] e = nativeRemoveLast(getNativeHandle());
    if (e == null) {
      throw new NoSuchElementException("List is empty");
    }
    return serializer.fromBytes(e);
  }

  @Override
  public void truncate(long newSize) {
    checkArgument(newSize >= 0, "New size must be non-negative: %s", newSize);
    notifyModified();
    nativeTruncate(getNativeHandle(), newSize);
  }

  @Override
  public final void clear() {
    notifyModified();
    nativeClear(getNativeHandle());
  }

  @Override
  public final boolean isEmpty() {
    return nativeIsEmpty(getNativeHandle());
  }

  @Override
  public final long size() {
    return nativeSize(getNativeHandle());
  }

  @Override
  public final Iterator<T> iterator() {
    return StorageIterators.createIterator(
        nativeCreateIter(getNativeHandle()),
        this::nativeIterNext,
        this::nativeIterFree,
        dbAccess,
        modCounter,
        serializer::fromBytes);
  }

  @Override
  public Stream<T> stream() {
    boolean immutable = !dbAccess.canModify();
    ListSpliterator<T> spliterator = new ListSpliterator<>(this, modCounter, immutable);
    return StreamSupport.stream(spliterator, false);
  }

  abstract void nativeAdd(long nativeHandle, byte[] e);

  abstract void nativeSet(long nativeHandle, long index, byte[] e);

  abstract byte[] nativeGet(long nativeHandle, long index);

  abstract byte[] nativeGetLast(long nativeHandle);

  abstract byte[] nativeRemoveLast(long nativeHandle);

  abstract void nativeTruncate(long nativeHandle, long newSize);

  abstract void nativeClear(long nativeHandle);

  abstract boolean nativeIsEmpty(long nativeHandle);

  abstract long nativeSize(long nativeHandle);

  abstract long nativeCreateIter(long nativeHandle);

  abstract byte[] nativeIterNext(long iterNativeHandle);

  abstract void nativeIterFree(long iterNativeHandle);
}
