package com.exceljava.jinx;

/**
 * <p>ExcelReference is used to fetch the data and metadata for a cell in Excel.</p>
 *
 * <p>Instances of ExcelReference are either passed to {@link com.exceljava.jinx.ExcelFunction} functions
 * as arguments, or returned from {@link com.exceljava.jinx.ExcelAddIn#getReference(IUnknown)}.</p>
 *
 * <p>Some of the properties of ExcelReference instances can only be accessed
 * if the calling function has been registered as a macro sheet equivalent function.</p>
 */
public interface ExcelReference {
    /**
     * ExcelRange specifies a rectangular region in an Excel sheet.
     *
     * Excel coordinates start from (0,0) at cell 'A1'.
     */
    interface Range {
        /**
         * Return the index of the first row in the range.
         */
        long getFirstRow();

        /**
         * Return the index of the last row in the range.
         */
        long getLastRow();

        /**
         * Return the index of the first column in the range.
         */
        long getFirstColumn();

        /**
         * Return the index of the last column in the range.
         */
        long getLastColumn();
    }

    /**
     * Return the value of the referenced cell.
     *
     * <p>
     *     Note: Must be called from a macro sheet equivalent function
     * </p>
     */
    Object getValue();

    /**
     * Return the value of the referenced cell as a specific type.
     *
     * <p>
     *     Note: Must be called from a macro sheet equivalent function
     * </p>
     */
    <T> T getValue(Class<T> cls);

    /**
     * Sets a value to the referenced range.
     *
     * This can only be called from a macro or menu function, or
     * by using ExcelAddIn.schedule.
     *
     * The value(s) will be converted automatically to Excel types,
     * and any unrecognized types will be cached in the object cache
     * with their handle(s) will be set on the Excel range.
     *
     * @param value Value to set on the range.
     */
    void setValue(Object value);

    /**
     * Return the address of the referenced cell.
     *
     * <p>
     *     Note: Must be called from a macro sheet equivalent function
     * </p>
     */
    String getAddress();

    /**
     * Return the formula of the referenced cell, or
     * null if the cell doesn't have a formula.
     *
     * <p>
     *     Note: Must be called from a macro sheet equivalent function
     * </p>
     */
    String getFormula();

    /**
     * Return the note set on the referenced cell,
     * or null if no note is present.
     *
     * <p>
     *     Note: Must be called from a macro sheet equivalent function
     * </p>
     */
    String getNote();

    /**
     * Return the name of the sheet this cell belongs to.
     */
    String getSheetName();

    /**
     * Return the internal id of the sheet this cell belongs to.
     */
    long getSheetId();

    /**
     * Return a value indicating whether the cell has been calculated or not.
     *
     * In almost all cases this will always be True as Excel will
     * automatically have recalculated the cell before passing it
     * to the function
     */
    boolean isCalculated();

    /**
     * Gets the coordinates of this cell or range of cells as a
     * 2d rectangle.
     *
     * Excel coordinates start from (0,0) at cell 'A1'.
     */
    Range getRange();
}
