/*
 * Copyright (C) 2017 - present by PyXLL Ltd.
 */
package com.exceljava.jinx;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * <p>ExcelFunction annotation for methods that should be exposed as Excel worksheet functions.</p>
 *
 * <p>{@link com.exceljava.jinx.ExcelArguments} and {@link com.exceljava.jinx.ExcelArgument}
 * can also be used to provide argument documentation.</p>
 *
 * <p>Static and non-static methods can be used. For non-static methods, the class must
 * have a public constructor taking either no parameters or a single {@link com.exceljava.jinx.ExcelAddIn}
 * parameter.</p>
 *
 * <p>Example:</p>
 *
 * <pre><code>
 * public class WorksheetFunctions {
 *     {@literal @}ExcelFunction(
 *            value = "jinx.multiplyAndAdd",
 *           description = "Simple function that returns (x * y) + z"
 *     )
 *     {@literal @}ExcelArguments({
 *          {@literal @}ExcelArgument("x"),
 *          {@literal @}ExcelArgument("y"),
 *          {@literal @}ExcelArgument("z")
 *     })
 *     public static double multiplyAndAdd(int x, double y, double z) {
 *          return (x * y) + z;
 *     }
 * }
 * </code></pre>
 */
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
public @interface ExcelFunction {

    /**
     * Name of the function as it will appear in Excel.
     */
    String value() default "";

    /**
     * Description shown in the Excel function wizard.
     */
    String description() default "";

    /**
     * Category the function will appear under in the Excel
     * function wizard.
     */
    String category() default "Jinx";

    /**
     * Not currently used.
     */
    String helpTopic() default "";

    /**
     * Volatile functions are calculated when Excel recalculates
     * regardless of whether any inputs have changed or not.
     */
    boolean isVolatile() default false;

    /**
     * Register the function as a macro sheet equivalent function.
     */
    boolean isMacroType() default false;

    /**
     * Thread safe functions may be run by Excel in threads other
     * than the main thread when it determines there may be a performance
     * benefit to doing so, depending on the number of calculation threads
     * available.
     */
    boolean isThreadSafe() default false;

    /**
     * Hidden functions do not appear in the Excel function wizard or
     * show up when using in-sheet IntelliSense.
     */
    boolean isHidden() default false;

    /**
     * Not currently used.
     */
    boolean allowAbort() default false;

    /**
     * <p>For functions returning arrays, if autoResize is true the formula
     * will automatically be resized to match the dimensions of the
     * returned array.</p>
     *
     * <p><em>Note:</em>RTD functions will not be automatically resized.
     * If you require an auto-resizing RTD function, have your RTD function
     * return an Object and use a second non-RTD function to expand the
     * result.</p>
     */
    boolean autoResize() default false;

    /**
     * Disable calling the method when being called from the Excel Function
     * Wizard.
     *
     * This is useful for slow calculations where calling the method from
     * the function wizard causes Excel to respond very slowly.
     */
    boolean disableInFunctionWizard() default false;

    /**
     * If the result of the function is an array, transpose the array before
     * returning to Excel.
     *
     * This can be used for returning a 1d array as a row instead of a
     * column, for example.
     */
    boolean transpose() default false;

    /**
     * <p>If true then the calling cell will be marked as dirty
     * when the workbook is opened. The next time Excel recalculates
     * then the cell will be recalculated, which will be immediately
     * on opening if automatic calculations are enabled.</p>
     *
     * <p>This can be useful for functions returning cached objects
     * to ensure that the object is created each time the workbook
     * is opened.</p>
     */
    boolean recalculateOnOpen() default false;
}
