package com.evolveasia.aws

import com.amazonaws.services.s3.AmazonS3Client
import com.amazonaws.services.s3.model.ObjectListing
import com.evolveasia.decodeSampledBitmapFromResource
import com.evolveasia.streamToByteArray
import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import io.reactivex.Single
import io.reactivex.android.schedulers.AndroidSchedulers
import java.io.ByteArrayOutputStream
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.OutputStream

fun uploadImageAWS(awsMetaInfo: AwsMetaInfo): Flowable<String> {
    return Flowable.create({ emitter ->
        awsMetaInfo.imageMetaInfo.imagePath = compressAwsImage(awsMetaInfo)
        amazonUploadSingle(awsMetaInfo)
                ?.subscribeOn(io.reactivex.schedulers.Schedulers.io())
                ?.observeOn(AndroidSchedulers.mainThread())
                ?.subscribe({
                    emitter.onComplete()
                }) {
                    it.printStackTrace()
                    emitter.onError(it)
                }

    }, BackpressureStrategy.LATEST)
}

//Warning amazon notifies on main thread
fun amazonUploadSingle(awsMetaInfo: AwsMetaInfo, uploadPath: String, maxFileSize: Long): Single<TransferUpdate?>? {
    return TransferObservable(awsMetaInfo, awsMetaInfo.imageMetaInfo.imagePath, uploadPath, maxFileSize)
            .doOnNext { transferUpdate ->
                if (transferUpdate.state == TransferUpdate.PROGRESSED_CHANGED)
                    println("Data upload Progress Progress:%d , Total:%d" + transferUpdate.byteCount + transferUpdate.byteTotal)
            }.lastOrError().observeOn(io.reactivex.schedulers.Schedulers.io())
}

fun amazonUploadSingle(awsMetaInfo: AwsMetaInfo): Single<TransferUpdate?>? {
    return amazonUploadSingle(awsMetaInfo, awsMetaInfo.awsFolderPath, TransferObservable.FILE_SIZE_UNLIMITED)
}

fun compressAwsImage(awsMetaInfo: AwsMetaInfo): String {

    try {
        val byteArray = streamToByteArray(FileInputStream(awsMetaInfo.imageMetaInfo.imagePath))
        val bitmap = decodeSampledBitmapFromResource(byteArray, awsMetaInfo.imageMetaInfo.imageWidth
                ?: Constants.DEFAULT_IMAGE_WIDTH, awsMetaInfo.imageMetaInfo.imageHeight
                ?: Constants.DEFAULT_IMAGE_HEIGHT, awsMetaInfo.imageMetaInfo.waterMarkInfo)

        val stream = ByteArrayOutputStream()
        bitmap.compress(awsMetaInfo.imageMetaInfo.compressFormat, awsMetaInfo.imageMetaInfo.compressLevel, stream)
        val os: OutputStream = FileOutputStream(awsMetaInfo.imageMetaInfo.imagePath)
        os.write(stream.toByteArray())
        os.close()
    } catch (e: Exception) {
        e.printStackTrace()
        awsMetaInfo.imageMetaInfo.imagePath
    }
    return awsMetaInfo.imageMetaInfo.imagePath
}

fun fetchAwsImage(serviceConfig: AwsMetaInfo.AWSConfig, queryPath: String): Single<List<String>> {
    return Single.fromCallable {
        return@fromCallable getImageFromS3Buckets(serviceConfig,queryPath)
    }

}


fun getImageFromS3Buckets(serviceConfig: AwsMetaInfo.AWSConfig, queryPath: String): List<String> {
    // Queries files in the bucket from S3.
    val s3 = Util.getS3Client(serviceConfig)
    var objectList = s3.listObjects(serviceConfig.bucketName,queryPath)
    val list = objectList.objectSummaries.map {
        s3.getUrl(serviceConfig.bucketName, it.key).toString()
    }.toMutableList()

    while (!objectList.isTruncated) {
        objectList = getObjectList(objectList, s3)
        list.addAll(objectList.objectSummaries.map {
            s3.getUrl(serviceConfig.bucketName, it.key).toString()
        })
    }
    return list
}

fun getObjectList(objectListing: ObjectListing, s3: AmazonS3Client): ObjectListing {
    return s3.listNextBatchOfObjects(objectListing)
}