package com.evolveasia.aws

import com.evolveasia.decodeSampledBitmapFromResource
import com.evolveasia.gCloudStorageUtils.Constants.DEFAULT_IMAGE_WIDTH
import com.evolveasia.streamToByteArray
import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import io.reactivex.Single
import io.reactivex.android.schedulers.AndroidSchedulers
import java.io.ByteArrayOutputStream
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.OutputStream

fun uploadImageAWS(gcsMetaInfo: AwsMetaInfo): Flowable<String> {
    return Flowable.create({ emitter ->
        gcsMetaInfo.imageMetaInfo.imagePath = compressAwsImage(gcsMetaInfo)
        amazonUploadSingle(gcsMetaInfo, gcsMetaInfo.awsFolerPath)
                ?.subscribeOn(io.reactivex.schedulers.Schedulers.io())
                ?.observeOn(AndroidSchedulers.mainThread())
                ?.subscribe({
                    emitter.onComplete()
                }) {
                    emitter.onError(it)
                }

    }, BackpressureStrategy.LATEST)
}

//Warning amazon notifies on main thread
fun amazonUploadSingle(awsMetaInfo: AwsMetaInfo, uploadPath: String?, maxFileSize: Long): Single<TransferUpdate?>? {
    return TransferObservable(awsMetaInfo, awsMetaInfo.imageMetaInfo.imagePath, uploadPath, maxFileSize)
            .doOnNext { transferUpdate ->
                if (transferUpdate.state == TransferUpdate.PROGRESSED_CHANGED)
                    println("Data upload Progress Progress:%d , Total:%d" + transferUpdate.byteCount + transferUpdate.byteTotal)
            }.lastOrError().observeOn(io.reactivex.schedulers.Schedulers.io())
}

fun amazonUploadSingle(awsMetaInfo: AwsMetaInfo, uploadPath: String?): Single<TransferUpdate?>? {
    return amazonUploadSingle(awsMetaInfo, uploadPath, TransferObservable.FILE_SIZE_UNLIMITED)
}

fun compressAwsImage(gcsMetaInfo: AwsMetaInfo): String {
    val byteArray = streamToByteArray(FileInputStream(gcsMetaInfo.imageMetaInfo.imagePath))
    val bitmap = decodeSampledBitmapFromResource(byteArray, gcsMetaInfo.imageMetaInfo.imageWidth
            ?: Constants.DEFAULT_IMAGE_WIDTH, gcsMetaInfo.imageMetaInfo.imageHeight
            ?: Constants.DEFAULT_IMAGE_HEIGHT, gcsMetaInfo.imageMetaInfo.waterMarkInfo)

    val stream = ByteArrayOutputStream()
    bitmap.compress(gcsMetaInfo.imageMetaInfo.compressFormat, gcsMetaInfo.imageMetaInfo.compressLevel, stream)
    try {
        val os: OutputStream = FileOutputStream(gcsMetaInfo.imageMetaInfo.imagePath)
        os.write(stream.toByteArray())
        os.close()
    } catch (e: Exception) {
        println("Exception: $e")
        gcsMetaInfo.imageMetaInfo.imagePath
    }
    return gcsMetaInfo.imageMetaInfo.imagePath
}