package com.evolveasia.gCloudStorageUtils

import com.evolveasia.gCloudStorageUtils.Constants.DEFAULT_IMAGE_HEIGHT
import com.evolveasia.gCloudStorageUtils.Constants.DEFAULT_IMAGE_WIDTH
import com.evolveasia.gCloudStorageUtils.Constants.TYPE_JSON
import com.google.api.services.storage.StorageScopes
import com.google.auth.oauth2.GoogleCredentials
import com.google.auth.oauth2.OAuth2Credentials
import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import io.reactivex.FlowableEmitter
import okhttp3.MediaType
import okhttp3.MultipartBody
import okhttp3.RequestBody
import org.json.JSONObject
import java.io.ByteArrayOutputStream
import java.io.File
import java.io.FileInputStream
import java.io.InputStream
import java.util.*

/**
 * @param [serviceConfig] : [InputStream] of the service key json
 * @return [OAuth2Credentials] for the storage app
 */
fun getAuthCredentials(serviceConfig: InputStream?): OAuth2Credentials {
    return GoogleCredentials.fromStream(serviceConfig)
            .createScoped(Collections.singleton(StorageScopes.DEVSTORAGE_FULL_CONTROL))
            .apply { refresh() }
}


fun getMetaDataRequestBody(gcsMetaInfo: GCSMetaInfo, emitter: FlowableEmitter<Double>): RequestBody {
    val jsonParams = JSONObject().apply {
        put("name", gcsMetaInfo.gcsStoragePath)
        if (gcsMetaInfo.imageMetaInfo.metadata.isEmpty()) return@apply
        put("metadata", gcsMetaInfo.imageMetaInfo.metadata)
    }
    return MultipartBody.Builder()
            .setType(MultipartBody.FORM)
            .addPart(RequestBody.create(TYPE_JSON, jsonParams.toString()))
            .addPart(RequestBody.create(MediaType.parse(gcsMetaInfo.imageMetaInfo.mediaType), gcsMetaInfo.imageMetaInfo.imageByteArray))
            .build()
}

//this is for with progress
//createCountingRequestBody(File(authInfo.content.name),emitter)
//this is for with out progress
//RequestBody.create(TYPE_CONTENT, authInfo.content.imageByteArray)


private fun createCountingRequestBody(file: File, emitter: FlowableEmitter<Double>): RequestBody {
    val requestBody = createRequestBody(file)
    println("create couting req body")
    print("image file name --> ${file.absolutePath}")
    return CountingRequestBody(requestBody) { bytesWritten, contentLength ->
        val progress: Double = (1.0 * bytesWritten) / contentLength
        println("prog: $progress")
        emitter.onNext(progress)
    }
}


private fun createRequestBody(file: File): RequestBody {
    return RequestBody.create(MediaType.parse(GCSMetaInfo.ImageMetaInfo.TYPE_JPEG), file)
}


fun uploadImageGCS(gcsMetaInfo: GCSMetaInfo): Flowable<Double> {
    return Flowable.create({ emitter ->
        gcsMetaInfo.imageMetaInfo.imageByteArray = compressImage(gcsMetaInfo.imageMetaInfo)
        val params = getMetaDataRequestBody(gcsMetaInfo, emitter)
        ApiModule.provideApiService(gcsMetaInfo).uploadImageToGCS(gcsMetaInfo.bucketName, params)
                .subscribe({
                    emitter.onComplete()
                }, {
                    emitter.onError(it)
                })
        // TODO: find solution for tryOnError i.e. emitter::tryOnError not working

    }, BackpressureStrategy.LATEST)
}

fun compressImage(imageMetaInfo: GCSMetaInfo.ImageMetaInfo): ByteArray {
    val bitmap = decodeSampledBitmapFromResource(streamToByteArray(FileInputStream(imageMetaInfo.imagePath)), imageMetaInfo.imageWidth
            ?: DEFAULT_IMAGE_WIDTH, imageMetaInfo.imageHeight ?: DEFAULT_IMAGE_HEIGHT)
    val stream = ByteArrayOutputStream()
    bitmap.compress(imageMetaInfo.compressFormat, imageMetaInfo.compressLevel, stream)
    return stream.toByteArray()
}
