/*
 * Decompiled with CFR 0.152.
 */
package com.evanlennick.retry4j;

import com.evanlennick.retry4j.AttemptResults;
import com.evanlennick.retry4j.CallResults;
import com.evanlennick.retry4j.config.RetryConfig;
import com.evanlennick.retry4j.config.RetryConfigBuilder;
import com.evanlennick.retry4j.exception.RetriesExhaustedException;
import com.evanlennick.retry4j.exception.UnexpectedException;
import com.evanlennick.retry4j.listener.AfterFailedTryListener;
import com.evanlennick.retry4j.listener.BeforeNextTryListener;
import com.evanlennick.retry4j.listener.OnCompletionListener;
import com.evanlennick.retry4j.listener.OnFailureListener;
import com.evanlennick.retry4j.listener.OnSuccessListener;
import com.evanlennick.retry4j.listener.RetryListener;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CallExecutor<T> {
    private Logger logger = LoggerFactory.getLogger(CallExecutor.class);
    private RetryConfig config;
    private AfterFailedTryListener afterFailedTryListener;
    private BeforeNextTryListener beforeNextTryListener;
    private OnFailureListener onFailureListener;
    private OnSuccessListener onSuccessListener;
    private OnCompletionListener onCompletionListener;
    private ExecutorService executorService;
    private Exception lastKnownExceptionThatCausedRetry;
    private CallResults<T> results = new CallResults();

    public CallExecutor() {
        this(new RetryConfigBuilder().fixedBackoff5Tries10Sec().build());
    }

    public CallExecutor(RetryConfig config) {
        this.config = config;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CallResults<T> execute(Callable<T> callable) throws RetriesExhaustedException, UnexpectedException {
        this.logger.trace("Starting retry4j execution with callable {}", (Object)this.config, callable);
        this.logger.debug("Starting retry4j execution with executor state {}", (Object)this);
        long start = System.currentTimeMillis();
        this.results.setStartTime(start);
        int maxTries = this.config.getMaxNumberOfTries();
        long millisBetweenTries = this.config.getDelayBetweenRetries().toMillis();
        this.results.setCallName(callable.toString());
        AttemptResults attemptResults = new AttemptResults();
        attemptResults.setSuccessful(false);
        try {
            int tries;
            for (tries = 0; tries < maxTries && !attemptResults.wasSuccessful(); ++tries) {
                this.logger.trace("Retry4j executing callable {}", callable);
                attemptResults = this.tryCall(callable);
                if (!attemptResults.wasSuccessful()) {
                    this.handleRetry(millisBetweenTries, tries + 1);
                }
                this.logger.trace("Retry4j retrying for time number {}", (Object)tries);
            }
            this.refreshRetryResults(attemptResults.wasSuccessful(), tries);
            this.results.setEndTime(System.currentTimeMillis());
            this.postExecutionCleanup(callable, maxTries, attemptResults);
            this.logger.debug("Finished retry4j execution in {} ms", (Object)this.results.getTotalElapsedDuration().toMillis());
            this.logger.trace("Finished retry4j execution with executor state {}", (Object)this);
        }
        finally {
            if (null != this.onCompletionListener) {
                this.onCompletionListener.onCompletion(this.results);
            }
        }
        return this.results;
    }

    public void executeAsync(Callable<T> callable) {
        if (null == this.executorService) {
            this.executorService = Executors.newFixedThreadPool(10);
        }
        Runnable runnable = () -> this.execute(callable);
        this.executorService.execute(runnable);
    }

    /*
     * Enabled aggressive block sorting
     */
    private void postExecutionCleanup(Callable<T> callable, int maxTries, AttemptResults<T> result) {
        if (result.wasSuccessful()) {
            this.results.setResult(result.getResult());
            if (null == this.onSuccessListener) return;
            this.onSuccessListener.onSuccess(this.results);
            return;
        }
        String failureMsg = String.format("Call '%s' failed after %d tries!", callable.toString(), maxTries);
        if (null != this.onFailureListener) {
            this.onFailureListener.onFailure(this.results);
            return;
        }
        this.logger.trace("Throwing retries exhausted exception");
        throw new RetriesExhaustedException(failureMsg, this.lastKnownExceptionThatCausedRetry, this.results);
    }

    private AttemptResults<T> tryCall(Callable<T> callable) throws UnexpectedException {
        AttemptResults<T> attemptResult = new AttemptResults<T>();
        try {
            T callResult = callable.call();
            attemptResult.setResult(callResult);
            attemptResult.setSuccessful(true);
            return attemptResult;
        }
        catch (Exception e) {
            if (this.shouldThrowException(e)) {
                this.logger.trace("Throwing expected exception {}", (Throwable)e);
                throw new UnexpectedException(e);
            }
            this.lastKnownExceptionThatCausedRetry = e;
            attemptResult.setSuccessful(false);
            return attemptResult;
        }
    }

    private void handleRetry(long millisBetweenTries, int tries) {
        this.refreshRetryResults(false, tries);
        if (null != this.afterFailedTryListener) {
            this.afterFailedTryListener.immediatelyAfterFailedTry(this.results);
        }
        this.sleep(millisBetweenTries, tries);
        if (null != this.beforeNextTryListener) {
            this.beforeNextTryListener.immediatelyBeforeNextTry(this.results);
        }
    }

    private void refreshRetryResults(boolean success, int tries) {
        long currentTime = System.currentTimeMillis();
        long elapsed = currentTime - this.results.getStartTime();
        this.results.setTotalTries(tries);
        this.results.setTotalElapsedDuration(Duration.of(elapsed, ChronoUnit.MILLIS));
        this.results.setSuccessful(success);
        this.results.setLastExceptionThatCausedRetry(this.lastKnownExceptionThatCausedRetry);
    }

    private void sleep(long millis, int tries) {
        Duration duration = Duration.of(millis, ChronoUnit.MILLIS);
        long millisToSleep = this.config.getBackoffStrategy().getMillisToWait(tries, duration);
        this.logger.trace("Retry4j executor sleeping for {} ms", (Object)millisToSleep);
        try {
            TimeUnit.MILLISECONDS.sleep(millisToSleep);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private boolean shouldThrowException(Exception e) {
        if (this.config.isRetryOnAnyException().booleanValue()) {
            return false;
        }
        for (Class<? extends Exception> exceptionInSet : this.config.getRetryOnSpecificExceptions()) {
            if (!exceptionInSet.isAssignableFrom(e.getClass())) continue;
            return false;
        }
        for (Class<? extends Exception> exceptionInSet : this.config.getRetryOnAnyExceptionExcluding()) {
            if (exceptionInSet.isAssignableFrom(e.getClass())) continue;
            return false;
        }
        return true;
    }

    public void registerRetryListener(RetryListener listener) {
        if (listener instanceof AfterFailedTryListener) {
            this.afterFailedTryListener = (AfterFailedTryListener)listener;
        } else if (listener instanceof BeforeNextTryListener) {
            this.beforeNextTryListener = (BeforeNextTryListener)listener;
        } else if (listener instanceof OnSuccessListener) {
            this.onSuccessListener = (OnSuccessListener)listener;
        } else if (listener instanceof OnFailureListener) {
            this.onFailureListener = (OnFailureListener)listener;
        } else if (listener instanceof OnCompletionListener) {
            this.onCompletionListener = (OnCompletionListener)listener;
        } else {
            throw new IllegalArgumentException("Tried to register an unrecognized RetryListener!");
        }
        this.logger.trace("Registered listener on retry4j executor {}", (Object)listener);
    }

    public void setConfig(RetryConfig config) {
        this.logger.trace("Set config on retry4j executor {}", (Object)config);
        this.config = config;
    }

    public void setExecutorService(ExecutorService executorService) {
        this.executorService = executorService;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("CallExecutor{");
        sb.append("config=").append(this.config);
        sb.append(", afterFailedTryListener=").append(this.afterFailedTryListener);
        sb.append(", beforeNextTryListener=").append(this.beforeNextTryListener);
        sb.append(", onFailureListener=").append(this.onFailureListener);
        sb.append(", onSuccessListener=").append(this.onSuccessListener);
        sb.append(", executorService=").append(this.executorService);
        sb.append(", lastKnownExceptionThatCausedRetry=").append(this.lastKnownExceptionThatCausedRetry);
        sb.append(", results=").append(this.results);
        sb.append('}');
        return sb.toString();
    }
}

