/*
 * Decompiled with CFR 0.152.
 */
package com.ethlo.time.internal;

import com.ethlo.time.DateTime;
import com.ethlo.time.Field;
import com.ethlo.time.ParseConfig;
import com.ethlo.time.TimezoneOffset;
import com.ethlo.time.internal.ErrorUtil;
import com.ethlo.time.internal.LimitedCharArrayIntegerUtil;
import java.time.OffsetDateTime;
import java.time.format.DateTimeParseException;
import java.util.Arrays;

public class ITUParser {
    public static final char DATE_SEPARATOR = '-';
    public static final char TIME_SEPARATOR = ':';
    public static final char SEPARATOR_UPPER = 'T';
    public static final char SEPARATOR_LOWER = 't';
    public static final char SEPARATOR_SPACE = ' ';
    static final char PLUS = '+';
    static final char MINUS = '-';
    public static final char FRACTION_SEPARATOR = '.';
    static final char ZULU_UPPER = 'Z';
    private static final char ZULU_LOWER = 'z';
    public static final int MAX_FRACTION_DIGITS = 9;

    private ITUParser() {
    }

    private static DateTime handleTime(int offset, ParseConfig parseConfig, String chars, int year, int month, int day, int hour, int minute) {
        switch (chars.charAt(offset + 16)) {
            case ':': {
                return ITUParser.handleTimeResolution(offset, parseConfig, year, month, day, hour, minute, chars);
            }
            case '+': 
            case '-': 
            case 'Z': 
            case 'z': {
                TimezoneOffset zoneOffset = ITUParser.parseTimezone(offset, parseConfig, chars, offset + 16);
                int charLength = Field.MINUTE.getRequiredLength() + (zoneOffset != null ? zoneOffset.getRequiredLength() : 0);
                return new DateTime(Field.MINUTE, year, month, day, hour, minute, 0, 0, zoneOffset, 0, charLength);
            }
        }
        throw ErrorUtil.raiseUnexpectedCharacter(chars, offset + 16);
    }

    private static void assertAllowedDateTimeSeparator(int offset, String chars, ParseConfig config) {
        int index = offset + 10;
        char needle = chars.charAt(index);
        if (!config.isDateTimeSeparator(needle)) {
            String allowedCharStr = config.getDateTimeSeparators().length > 1 ? Arrays.toString(config.getDateTimeSeparators()) : Character.toString(config.getDateTimeSeparators()[0]);
            throw new DateTimeParseException(String.format("Expected character %s at position %d, found %s: %s", allowedCharStr, index + 1, Character.valueOf(chars.charAt(index)), chars), chars, index);
        }
    }

    private static TimezoneOffset parseTimezone(int offset, ParseConfig parseConfig, String chars, int idx) {
        if (idx >= chars.length()) {
            return null;
        }
        int len = chars.length();
        int left = len - idx;
        char c = chars.charAt(idx);
        if (c == 'Z' || c == 'z') {
            ITUParser.assertNoMoreChars(offset, parseConfig, chars, idx);
            return TimezoneOffset.UTC;
        }
        char sign = chars.charAt(idx);
        if (sign != '+' && sign != '-') {
            ErrorUtil.raiseUnexpectedCharacter(chars, idx);
        }
        if (left < 6) {
            throw new DateTimeParseException(String.format("Invalid timezone offset: %s", chars), chars, idx);
        }
        int hours = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, idx + 1, idx + 3);
        int minutes = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, idx + 4, idx + 4 + 2);
        if (sign == '-') {
            hours = -hours;
            minutes = -minutes;
            if (hours == 0 && minutes == 0) {
                throw new DateTimeParseException("Unknown 'Local Offset Convention' date-time not allowed", chars, idx);
            }
        }
        ITUParser.assertNoMoreChars(offset, parseConfig, chars, idx + 6);
        return TimezoneOffset.ofHoursMinutes(hours, minutes);
    }

    private static void assertNoMoreChars(int offset, ParseConfig parseConfig, String chars, int lastUsed) {
        if (parseConfig.isFailOnTrailingJunk() && offset == 0 && chars.length() > lastUsed + 1) {
            throw new DateTimeParseException(String.format("Trailing junk data after position %d: %s", lastUsed + 2, chars), chars, lastUsed + 1);
        }
    }

    public static DateTime parseLenient(String chars, ParseConfig parseConfig, int offset) {
        if (chars == null) {
            throw new NullPointerException("text cannot be null");
        }
        int availableLength = chars.length() - offset;
        if (availableLength < 0) {
            throw new IndexOutOfBoundsException(String.format("offset is %d which is equal to or larger than the input length of %d", offset, chars.length()));
        }
        if (offset < 0) {
            throw new IndexOutOfBoundsException(String.format("offset cannot be negative, was %d", offset));
        }
        int years = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset, offset + 4);
        if (4 == availableLength) {
            return DateTime.ofYear(years);
        }
        ErrorUtil.assertPositionContains(Field.MONTH, chars, offset + 4, '-');
        int months = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset + 5, offset + 7);
        if (7 == availableLength) {
            return DateTime.ofYearMonth(years, months);
        }
        ErrorUtil.assertPositionContains(Field.DAY, chars, offset + 7, '-');
        int days = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset + 8, offset + 10);
        if (10 == availableLength) {
            return DateTime.ofDate(years, months, days);
        }
        ITUParser.assertAllowedDateTimeSeparator(offset, chars, parseConfig);
        int hours = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset + 11, offset + 13);
        ErrorUtil.assertPositionContains(Field.MINUTE, chars, offset + 13, ':');
        int minutes = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset + 14, offset + 16);
        if (availableLength == 16) {
            return DateTime.of(years, months, days, hours, minutes, null);
        }
        return ITUParser.handleTime(offset, parseConfig, chars, years, months, days, hours, minutes);
    }

    private static DateTime handleTimeResolution(int offset, ParseConfig parseConfig, int year, int month, int day, int hour, int minute, String chars) {
        int length = chars.length() - offset;
        if (length > 19) {
            char c = chars.charAt(offset + 19);
            if (parseConfig.isFractionSeparator(c)) {
                return ITUParser.handleFractionalSeconds(offset, parseConfig, year, month, day, hour, minute, chars);
            }
            if (c == 'Z' || c == 'z') {
                TimezoneOffset timezoneOffset = TimezoneOffset.UTC;
                return ITUParser.handleSecondResolution(offset, year, month, day, hour, minute, chars, timezoneOffset);
            }
            if (c == '+' || c == '-') {
                TimezoneOffset timezoneOffset = ITUParser.parseTimezone(offset, parseConfig, chars, offset + 19);
                return ITUParser.handleSecondResolution(offset, year, month, day, hour, minute, chars, timezoneOffset);
            }
            throw ErrorUtil.raiseUnexpectedCharacter(chars, offset + 19);
        }
        if (length == offset + 19) {
            int seconds = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset + 17, offset + 19);
            return DateTime.of(year, month, day, hour, minute, seconds, null);
        }
        throw ErrorUtil.raiseUnexpectedEndOfText(chars, offset + 16);
    }

    private static DateTime handleSecondResolution(int offset, int year, int month, int day, int hour, int minute, String chars, TimezoneOffset timezoneOffset) {
        int seconds = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset + 17, offset + 19);
        int charLength = Field.SECOND.getRequiredLength() + (timezoneOffset != null ? timezoneOffset.getRequiredLength() : 0);
        return new DateTime(Field.SECOND, year, month, day, hour, minute, seconds, 0, timezoneOffset, 0, charLength);
    }

    private static DateTime handleFractionalSeconds(int offset, ParseConfig parseConfig, int year, int month, int day, int hour, int minute, String chars) {
        char c;
        int idx;
        int fractionDigits = 0;
        int nanos = 0;
        for (idx = offset + 20; idx < chars.length() && (c = chars.charAt(idx)) >= '0' && c <= '9'; ++idx) {
            ++fractionDigits;
            nanos = nanos * 10 + (c - 48);
        }
        ErrorUtil.assertFractionDigits(chars, fractionDigits, offset + (idx - 1));
        for (int pos = fractionDigits; pos < 9; ++pos) {
            nanos *= 10;
        }
        TimezoneOffset timezoneOffset = ITUParser.parseTimezone(offset, parseConfig, chars, idx);
        int charLength = idx + (timezoneOffset != null ? timezoneOffset.getRequiredLength() : 0) - offset;
        int second = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset + 17, offset + 19);
        return new DateTime(Field.NANO, year, month, day, hour, minute, second, nanos, timezoneOffset, fractionDigits, charLength);
    }

    public static OffsetDateTime parseDateTime(String chars, int offset) {
        DateTime dateTime = ITUParser.parseLenient(chars, ParseConfig.DEFAULT, offset);
        if (dateTime.includesGranularity(Field.SECOND)) {
            return dateTime.toOffsetDatetime();
        }
        Field field = dateTime.getMostGranularField();
        Field nextGranularity = Field.values()[field.ordinal() + 1];
        throw new DateTimeParseException(String.format("Unexpected end of input, missing field %s: %s", new Object[]{nextGranularity, chars}), chars, field.getRequiredLength());
    }
}

