package com.estimote.scanning_sdk.packet_provider.parsers

import android.annotation.TargetApi
import android.os.Build
import android.os.ParcelUuid
import com.estimote.scanning_sdk.common.toHex
import com.estimote.scanning_sdk.packet_provider.EstimoteLocationPacket
import com.estimote.scanning_sdk.packet_provider.EstimoteMacAddress
import com.estimote.scanning_sdk.packet_provider.scanner.EstimoteScanResult
import com.estimote.scanning_sdk.packet_provider.use_cases.EstimoteScanResultParser
import java.nio.ByteBuffer
import java.util.*

/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class EstimoteLocationV1Parser : EstimoteScanResultParser<EstimoteLocationPacket> {

    private val ESTIMOTE_SERVICE_UUID = ParcelUuid(UUID.fromString("0000fe9a-0000-1000-8000-00805f9b34fb"))

    override fun parse(result: EstimoteScanResult): EstimoteLocationPacket {
        val bytes = ByteBuffer.wrap(result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID))
        with(bytes) {
            return EstimoteLocationPacket(
                    getDeviceId(),
                    getChannel(),
                    getProtocolVersion(),
                    getMeasuredPower(),
                    EstimoteMacAddress(result.device.address),
                    result.rssi,
                    result.timestampNanosSinceBoot
            )
        }
    }

    private fun ByteBuffer.getDeviceId(): String {
        val idBytes = ByteArray(16)
        position(1)
        get(idBytes, 0, 16)
        return idBytes.toHex()
    }

    private fun ByteBuffer.getChannel(): Int {
        val channelValue = get(18).toInt() and 0x3
        return decodeChannelValue(channelValue)
    }

    private fun ByteBuffer.getMeasuredPower(): Int = this.get(17).toInt()

    private fun ByteBuffer.getProtocolVersion(): Int = this.get(0).toInt() and 0xF0 shr 4

    private fun decodeChannelValue(value: Int) = when (value) {
        0 -> 0
        1 -> 37
        2 -> 38
        3 -> 39
        else -> throw IllegalArgumentException("Wrong estimote location channel value: $value")
    }

}