package com.estimote.scanning_sdk.packet_provider.parsers

import android.annotation.TargetApi
import android.os.Build
import android.os.ParcelUuid
import com.estimote.internal_plugins_api.scanning.Duration
import com.estimote.internal_plugins_api.scanning.Magnetometer
import com.estimote.scanning_sdk.common.toHex
import com.estimote.scanning_sdk.common.toUnsignedInt
import com.estimote.scanning_sdk.packet_provider.EstimoteMacAddress
import com.estimote.scanning_sdk.packet_provider.EstimoteMagnetometer
import com.estimote.scanning_sdk.packet_provider.EstimoteTelemetryFrameBPacket
import com.estimote.scanning_sdk.packet_provider.EstimoteUptimeDuration
import com.estimote.scanning_sdk.packet_provider.scanner.EstimoteScanResult
import com.estimote.scanning_sdk.packet_provider.use_cases.EstimoteScanResultParser
import java.nio.ByteBuffer
import java.util.*
import java.util.concurrent.TimeUnit
import kotlin.experimental.and

/**
 * @author arek.biela@estimote.com (Arek Biela).
 */

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class EstimoteTelemetryBV1Parser : EstimoteScanResultParser<EstimoteTelemetryFrameBPacket> {

    private val ESTIMOTE_SERVICE_UUID = ParcelUuid(UUID.fromString("0000fe9a-0000-1000-8000-00805f9b34fb"))

    override fun parse(result: EstimoteScanResult): EstimoteTelemetryFrameBPacket {
        val bytes = ByteBuffer.wrap(result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID))
        with (bytes) {
            return EstimoteTelemetryFrameBPacket(
                    EstimoteMacAddress(result.device.address),
                    result.rssi,
                    result.timestampNanosSinceBoot,
                    getShortIdentifier(),
                    getMagnetometer(),
                    getAmbientLight(),
                    getUptime(),
                    getTemperature(),
                    getBatteryVoltage(),
                    getBatteryLevel()
            )
        }
    }

    private fun ByteBuffer.getShortIdentifier(): String {
        val shortIdBytes = ByteArray(8)
        this.position(1)
        this.get(shortIdBytes, 0, 8)
        return shortIdBytes.toHex()
    }

    private fun ByteBuffer.getMagnetometer(): Magnetometer {
        val x = get(10) / 128.0
        val y = get(11) / 128.0
        val z = get(12) / 128.0
        return EstimoteMagnetometer(x, y, z)
    }

    private fun ByteBuffer.getAmbientLight(): Double {
        val ambientByte = get(13)
        return Math.pow(2.0, (ambientByte.and(0b11110000.toByte()).toUnsignedInt() shr 4).toDouble()) * ambientByte.and(0b00001111).toUnsignedInt().toDouble() * 0.72
    }

    private fun ByteBuffer.getUptime(): Duration {
        val uptimeLowerByte = get(14)
        val uptimeHigherByte = get(15).and(0b00111111.toByte())
        val durationValue = uptimeLowerByte.toUnsignedInt() + (uptimeHigherByte.and(0b00001111).toUnsignedInt() shl 8)
        val unitValue = transformToTimeUnit(uptimeHigherByte.toUnsignedInt() shr 4)
        val recalculatedValue = recalculateIfCountedInWeeks(durationValue, unitValue)
        return EstimoteUptimeDuration(recalculatedValue, unitValue)
    }

    private fun ByteBuffer.getTemperature(): Double {
        val temperatureLowByte = get(15).and(0b11000000.toByte())
        val temperatureMidByte = get(16)
        val temperatureHighByte = get(17).and(0b00000011.toByte())
        return ((temperatureLowByte.toUnsignedInt() shr 6) + (temperatureMidByte.toUnsignedInt() shl 2) + (temperatureHighByte.toUnsignedInt() shl 10)) / 16.0
    }

    private fun ByteBuffer.getBatteryVoltage(): Int {
        val batteryVoltageLowByte = get(17).and(0b11111100.toByte())
        val batteryVoltageHighByte = get(18)
        return (batteryVoltageLowByte.toUnsignedInt() shr 2) + (batteryVoltageHighByte.toUnsignedInt() shl 6)
    }

    private fun ByteBuffer.getBatteryLevel(): Int {
        return get(19).toUnsignedInt()
    }

    private fun transformToTimeUnit(value: Int): TimeUnit {
        return when (value) {
            0 -> TimeUnit.SECONDS
            1 -> TimeUnit.MINUTES
            2 -> TimeUnit.HOURS
            3 -> TimeUnit.DAYS
            else -> TimeUnit.MILLISECONDS
        }
    }

    private fun recalculateIfCountedInWeeks(value: Int, timeUnit: TimeUnit): Int {
        return if (timeUnit == TimeUnit.DAYS && value > 31) 7 * value else value
    }
}