package com.estimote.scanning_sdk.packet_provider.use_cases

import android.annotation.TargetApi
import android.os.Build
import com.estimote.scanning_sdk.packet_provider.BeaconPacket
import com.estimote.scanning_sdk.packet_provider.antenna_bias.AntennaBiasCorrection
import com.estimote.scanning_sdk.packet_provider.rssi_smoothers.RssiSmoother
import com.estimote.scanning_sdk.settings.IBeaconScanSettings
import io.reactivex.Observable

/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class IBeaconScanUseCase (private val estimoteScanner: EstimoteScanner,
                                   private val recognizer: EstimoteScanResultRecognizer,
                                   private val parser: EstimoteScanResultParser<BeaconPacket>,
                                   private val rssiSmootherCache: ExpirationCache<String, RssiSmoother>,
                                   private val antennaBiasCorrection: AntennaBiasCorrection){


    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    fun run(scanSettings: IBeaconScanSettings): Observable<BeaconPacket> {
        return estimoteScanner.scan(scanSettings)
                .filter { recognizer.recognize(it) }
                .map { parser.parse(it) }
                .map { it.smoothRssiAndApplyAntennaBiasCorrection() }
    }

    private fun BeaconPacket.smoothRssiAndApplyAntennaBiasCorrection(): BeaconPacket {
        val smoother = rssiSmootherCache.get(macAddress.address)
        val smoothedRssi = smoother.smooth(rssi, this.timestamp/1000000000.0)
        return this.copy(rssi = smoothedRssi.toInt(), measuredPower = (measuredPower + antennaBiasCorrection.correction))
    }
}

