package com.estimote.scanning_sdk.packet_provider.parsers

import android.annotation.TargetApi
import android.os.Build
import com.estimote.scanning_sdk.common.toHex
import com.estimote.scanning_sdk.packet_provider.EstimoteMacAddress
import com.estimote.scanning_sdk.packet_provider.EstimoteNearablePacket
import com.estimote.scanning_sdk.packet_provider.scanner.EstimoteScanResult
import com.estimote.scanning_sdk.packet_provider.use_cases.EstimoteScanResultParser
import java.nio.ByteBuffer
import java.nio.ByteOrder
import kotlin.experimental.and

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class EstimoteNearableParser : EstimoteScanResultParser<EstimoteNearablePacket> {

    private val ESTIMOTE_MANUFACTURER_ID = 0x015D

    override fun parse(result: EstimoteScanResult): EstimoteNearablePacket {
        val bytes = ByteBuffer.wrap(result.scanRecord.getManufacturerSpecificData(ESTIMOTE_MANUFACTURER_ID))
        with(bytes) {
            return EstimoteNearablePacket(
                    getDeviceId(),
                    getHardwareVersion(),
                    getSoftwareVersion(),
                    getTemperature(),
                    getBatteryVoltage(),
                    isMoving(),
                    getXAcceleration(),
                    getYAcceleration(),
                    getZAcceleration(),
                    getCurrentMotionStateSeconds(),
                    getPreviousMotionStateSeconds(),
                    getTxPower(),
                    getAdvertisingChannel(),
                    EstimoteMacAddress(result.device.address),
                    result.rssi,
                    result.timestampNanosSinceBoot
            )
        }
    }

    private fun ByteBuffer.getDeviceId(): String {
        val bytes = ByteArray(8)
        position(1)
        get(bytes, 0, 8)
        return bytes.toHex()
    }

    private fun ByteBuffer.getHardwareVersion(): String {
        position(9)
        return when (get()) {
            0x00.toByte() -> "Hardware unknown"
            0x01.toByte() -> "D3.2"
            0x02.toByte() -> "D3.3"
            0x03.toByte() -> "D3.4"
            0x04.toByte() -> "SB0"
            0x05.toByte() -> "Sb4b"
            0x06.toByte() -> "SB5b"
            0x07.toByte() -> "S3A6"
            0x08.toByte() -> "S3A6b"
            0x09.toByte() -> "SCA2"
            0x0A.toByte() -> "SCA3"
            0x0B.toByte() -> "SC1.0"
            else -> "RFU"
        }
    }

    private fun ByteBuffer.getSoftwareVersion(): String {
        position(10)
        return when(get()) {
            0x00.toByte() -> "Bootloader unknown"
            0x01.toByte() -> "Bootloader 1.0.0"
            0x80.toByte() -> "Main app unknown"
            0x81.toByte() -> "Main app 1.0.0"
            0x82.toByte() -> "Main app 1.0.1"
            0x83.toByte() -> "Main app 1.1.0"
            0x84.toByte() -> "Main app 1.2.0"
            0x85.toByte() -> "Main app 1.3.0"
            else -> "RFU"
        }
    }

    private fun ByteBuffer.getTemperature(): Double {
        val bytes = ByteArray(2)
        position(11)
        get(bytes, 0, 2)
        return ((ByteBuffer.wrap(bytes).order(ByteOrder.LITTLE_ENDIAN).short and 0x0FFF).toInt() shl 4) / 256.0
    }

    private fun ByteBuffer.getBatteryVoltage(): Double {
        val bytes = ByteArray(2)
        position(12)
        get(bytes, 0, 2)
        val bat = ((bytes[0].toInt() and 0xFF) shr 4) + ((bytes[1].toInt() and 0x3F) shl 4)
        return (3 * 1.2 * bat) / 1023
    }

    private fun ByteBuffer.isMoving(): Boolean {
        position(13)
        return get() and 0x40 > 0
    }

    private fun ByteBuffer.getXAcceleration(): Double {
        position(14)
        return get() * 0.015625
    }

    private fun ByteBuffer.getYAcceleration(): Double {
        position(15)
        return get() * 0.015625
    }

    private fun ByteBuffer.getZAcceleration(): Double {
        position(16)
        return get() * 0.015625
    }

    private fun ByteBuffer.getCurrentMotionStateSeconds(): Int {
        position(17)
        val byte = get()
        val unit = (byte.toInt() and 0xC0) shr 6
        val value = (byte.toInt() and 0x3F)
        return when(unit){
            0x01 -> 60 * value
            0x02 -> 3600 * value
            0x03 -> if(value < 32) 86400 * value else 604800 * (value - 32)
            else -> value
        }
    }

    private fun ByteBuffer.getPreviousMotionStateSeconds(): Int {
        position(18)
        val byte = get()
        val unit = (byte.toInt() and 0xC0) shr 6
        val value = (byte.toInt() and 0x3F)
        return when(unit){
            0x01 -> 60 * value
            0x02 -> 3600 * value
            0x03 -> if(value < 32) 86400 * value else 604800 * (value - 32)
            else -> value
        }
    }

    private fun ByteBuffer.getTxPower(): Byte {
        position(19)
        return when((get() and 0x0F)) {
            0x00.toByte() -> -30
            0x01.toByte() -> -20
            0x02.toByte() -> -16
            0x03.toByte() -> -12
            0x04.toByte() -> -8
            0x05.toByte() -> -4
            0x06.toByte() -> -0
            else -> 4
        }
    }

    private fun ByteBuffer.getAdvertisingChannel(): Byte {
        position(19)
        return when((get() and 0x30).toInt() shr 4) {
            0x01 -> 37
            0x02 -> 38
            0x03 -> 39
            else -> -1
        }
    }
}