package com.estimote.scanning_sdk.packet_provider.parsers

import android.annotation.TargetApi
import android.os.Build
import android.os.ParcelUuid
import com.estimote.scanning_sdk.common.toHex
import com.estimote.scanning_sdk.packet_provider.EstimoteMacAddress
import com.estimote.scanning_sdk.packet_provider.EstimoteMeshPacket
import com.estimote.scanning_sdk.packet_provider.scanner.EstimoteScanResult
import com.estimote.scanning_sdk.packet_provider.use_cases.EstimoteScanResultParser
import java.nio.ByteBuffer
import java.nio.ByteOrder
import java.util.*
import kotlin.experimental.and

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class EstimoteMeshParser : EstimoteScanResultParser<EstimoteMeshPacket> {

    private val ESTIMOTE_SERVICE_UUID = ParcelUuid(UUID.fromString("0000fe9a-0000-1000-8000-00805f9b34fb"))

    override fun parse(result: EstimoteScanResult): EstimoteMeshPacket {
        val bytes = ByteBuffer.wrap(result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID))
        with(bytes) {
            return EstimoteMeshPacket(
                    getDeviceId(),
                    getReportHash(),
                    getMeshId(),
                    getSettingsVersion(),
                    getAppVersion(),
                    getBootloaderVersion(),
                    EstimoteMacAddress(result.device.address),
                    result.rssi,
                    result.timestampNanosSinceBoot
            )
        }
    }

    private fun ByteBuffer.getDeviceId(): String {
        val bytes = ByteArray(4)
        position(1)
        get(bytes, 0, 4)
        return bytes.toHex()
    }

    private fun ByteBuffer.getReportHash(): Int {
        val bytes = ByteArray(4)
        position(5)
        get(bytes, 0, 4)
        return ByteBuffer.wrap(bytes).order(ByteOrder.LITTLE_ENDIAN).int
    }

    private fun ByteBuffer.getMeshId(): Int {
        val bytes = ByteArray(4)
        position(9)
        get(bytes, 0, 4)
        return ByteBuffer.wrap(bytes).order(ByteOrder.LITTLE_ENDIAN).int
    }

    private fun ByteBuffer.getSettingsVersion(): Int {
        val bytes = ByteArray(4)
        position(13)
        get(bytes, 0, 4)
        return ByteBuffer.wrap(bytes).order(ByteOrder.LITTLE_ENDIAN).int
    }

    private fun ByteBuffer.getAppVersion(): String {
        val bytes = ByteArray(2)
        position(17)
        get(bytes, 0, 2)
        return "${bytes[1].toVersionOctet()}.${(bytes[0].toInt() shr 4).toVersionOctet()}.${bytes[0].toVersionOctet()}"
    }

    private fun ByteBuffer.getBootloaderVersion(): String {
        val bytes = ByteArray(2)
        position(18)
        get(bytes, 0, 2)
        return "${(bytes[1].toInt() shr 4).toVersionOctet()}.${bytes[1].toVersionOctet()}.${(bytes[0].toInt() shr 4).toVersionOctet()}"
    }

    private fun Byte.toVersionOctet() =
        (this and 0x0F).toString()

    private fun Int.toVersionOctet() =
            (this and 0x0F).toString()



}