package com.estimote.scanning_sdk.packet_provider.telemetry_merger

import com.estimote.android_ketchup.rx_goodness.reduceUntil
import com.estimote.internal_plugins_api.scanning.*
import com.estimote.scanning_sdk.packet_provider.EstimoteTelemetryFullPacket
import io.reactivex.Observable
import io.reactivex.observables.GroupedObservable

/**
 * @author Estimote Inc. (contact@estimote.com)
 */
internal class RxTelemetryMerger : TelemetryMerger {

    private data class TelemetryTriple(val frameA: EstimoteTelemetryFrameA?,
                                       val frameB: EstimoteTelemetryFrameB?,
                                       val connectivityFrame: EstimoteConnectivity?)

    override fun mergedTelemetryStream(scanA: Observable<out EstimoteTelemetryFrameA>,
                                       scanB: Observable<out EstimoteTelemetryFrameB>,
                                       connectivityScan: Observable<out EstimoteConnectivity>): Observable<EstimoteTelemetryFull> =
            mergeTelemetryScansAndConnectivityIntoOnePacketStream(scanA, scanB, connectivityScan)
                    .groupFramesByShortId()
                    .bufferTelemetryFramesForEachMacAddress()
                    .mergeFramesToFullTelemetryFrame()


    private fun Observable<Packet>.groupFramesByShortId() = groupBy {
        when (it) {
            is EstimoteTelemetryFrameA -> it.shortId
            is EstimoteTelemetryFrameB -> it.shortId
            is EstimoteConnectivity -> it.deviceId.substring(0, it.deviceId.length / 2)
            else -> throw IllegalArgumentException("Unsupported packet type. Only Estimote Telemetry or Estimote Connectivity is allowed.")
        }
    }

    private fun Observable<GroupedObservable<String, Packet>>.bufferTelemetryFramesForEachMacAddress() =
            flatMap {
                it.reduceUntil(
                        { TelemetryTriple(null, null, null) },
                        { packet, buffer ->
                            when (packet) {
                                is EstimoteTelemetryFrameA -> TelemetryTriple(packet, buffer.frameB, buffer.connectivityFrame)
                                is EstimoteTelemetryFrameB -> TelemetryTriple(buffer.frameA, packet, buffer.connectivityFrame)
                                is EstimoteConnectivity -> TelemetryTriple(buffer.frameA, buffer.frameB, packet)
                                else -> buffer
                            }
                        },
                        { it.frameA != null && it.frameB != null && it.connectivityFrame != null })
            }

    private fun mergeTelemetryScansAndConnectivityIntoOnePacketStream(scanA: Observable<out EstimoteTelemetryFrameA>,
                                                                      scanB: Observable<out EstimoteTelemetryFrameB>,
                                                                      connectivityScan: Observable<out EstimoteConnectivity>) =
            scanA.map { it as Packet }.mergeWith(scanB.map { it as Packet }).mergeWith(connectivityScan.map { it as Packet })


    private fun Observable<TelemetryTriple>.mergeFramesToFullTelemetryFrame() =
            map {
                EstimoteTelemetryFullPacket(
                        it.frameA!!.macAddress,
                        it.frameA.rssi,
                        maxOf(it.frameA.timestamp, it.frameB!!.timestamp),
                        it.connectivityFrame!!.deviceId,
                        it.frameB.magnetometer,
                        it.frameB.ambientLightInLux,
                        it.frameB.uptime,
                        it.frameB.temperatureInCelsiusDegrees,
                        it.frameB.batteryVoltageInMilliVolts,
                        it.frameB.batteryLevelPercentage,
                        it.frameA.acceleration,
                        it.frameA.motionState,
                        it.frameA.currentMotionDuration,
                        it.frameA.previousMotionDuration,
                        it.frameA.gpio,
                        it.frameA.pressure) as EstimoteTelemetryFull
            }


}