package com.estimote.scanning_sdk.packet_provider.parsers

import android.annotation.TargetApi
import android.os.Build
import android.os.ParcelUuid
import com.estimote.scanning_sdk.common.toHex
import com.estimote.scanning_sdk.packet_provider.EstimoteMacAddress
import com.estimote.scanning_sdk.packet_provider.EstimoteUwbPacket
import com.estimote.scanning_sdk.packet_provider.scanner.EstimoteScanResult
import com.estimote.scanning_sdk.packet_provider.use_cases.EstimoteScanResultParser
import java.nio.ByteBuffer
import java.util.*
import kotlin.collections.HashMap
import kotlin.experimental.and

/**
 * @author arek.biela@estimote.com (Arek Biela).
 */


@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class EstimoteUwbParser : EstimoteScanResultParser<EstimoteUwbPacket> {

    private val ESTIMOTE_SERVICE_UUID = ParcelUuid(UUID.fromString("0000fe9a-0000-1000-8000-00805f9b34fb"))
    private val DISTANCE_ESTIMATE_LENGTH = 5
    private val DISTANCE_ESTIMATE_OFFSET = 4

    override fun parse(result: EstimoteScanResult): EstimoteUwbPacket {
        val bytes = ByteBuffer.wrap(result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID))
        with(bytes) {
            return EstimoteUwbPacket(
                    getDeviceId(),
                    getDistanceEstimatesMap(),
                    getRangingInProgress(),
                    EstimoteMacAddress(result.device.address),
                    result.rssi,
                    result.timestampNanosSinceBoot)
        }
    }

    private fun ByteBuffer.getDeviceId(): String {
        val idBytes = ByteArray(3)
        position(1)
        get(idBytes, 0, 3)
        return idBytes.toHex()
    }

    private fun ByteBuffer.getDistanceEstimatesMap(): Map<String, Float> {

        val distanceEstimates = HashMap<String, Float>()
        for (i in 0..2) {
            val idBytes = ByteArray(3)
            position(i * DISTANCE_ESTIMATE_LENGTH + DISTANCE_ESTIMATE_OFFSET)
            get (idBytes, 0, 3)
            val identifier = idBytes.toHex()

            val distanceBytes = ByteArray(2)
            get(distanceBytes, 0, 2)
            val distanceEstimate = distanceBytes.getDistanceEstimate()

            distanceEstimates.put(identifier, distanceEstimate)
        }
        return distanceEstimates
    }

    private fun ByteArray.getDistanceEstimate(): Float {
        val reversedDistanceBytes = reversedArray()
        return reversedDistanceBytes.toHex().toLong(radix = 16) / 256.0f
    }

    private fun ByteBuffer.getRangingInProgress(): Boolean {
        return get(19).and(0x01) > 0
    }
}