package com.estimote.scanning_plugin.packet_provider.scanner.settings_transformers

import android.annotation.TargetApi
import android.bluetooth.le.ScanSettings
import android.os.Build
import com.estimote.scanning_plugin.settings.EstimoteScanSettings.ScanCallbackMode
import com.estimote.scanning_plugin.settings.EstimoteScanSettings.ScanPowerMode
import com.estimote.scanning_plugin.settings.ScanSettingsVisitor

internal class ScanSettingsTransformerProvider(private val androidApiVersion: Int,
                                               private val manufacturerId: String,
                                               private val isOffloadedScanBatchingSupported: Boolean) {

    fun getTransformer(): ScanSettingsVisitor<ScanSettings> {
        val powerModeTransformer =
                if (isOffloadedScanBatchingSupported)
                    getManufacturerSpecificTransformer() else defaultPowerModeTransformer()
        val callbackModeTransformer =
                if (androidApiVersion >= Build.VERSION_CODES.M)
                    marshmallowCallbackModeTransformer() else lollipopCallbackModeTransformer()
        return DefaultScanSettingsTransformer(powerModeTransformer, callbackModeTransformer)
    }

    private fun getManufacturerSpecificTransformer() = when {
        manufacturerId.isSamsungDevice() -> samsungPowerModeTransformer()
        else -> defaultPowerModeTransformer()
    }

    private fun String.isSamsungDevice() = this.equals("samsung", ignoreCase = true)

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    private fun defaultPowerModeTransformer(): (ScanSettings.Builder, ScanPowerMode) -> ScanSettings.Builder = { builder, powerMode ->
        when (powerMode) {
            ScanPowerMode.LOW_LATENCY -> builder.setReportDelay(0).setScanMode(ScanSettings.SCAN_MODE_LOW_LATENCY)
            ScanPowerMode.BALANCED -> builder.setReportDelay(0).setScanMode(ScanSettings.SCAN_MODE_BALANCED)
            ScanPowerMode.LOW_POWER -> builder.setReportDelay(0).setScanMode(ScanSettings.SCAN_MODE_LOW_POWER)
        }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    private fun samsungPowerModeTransformer(): (ScanSettings.Builder, ScanPowerMode) -> ScanSettings.Builder = { builder, powerMode ->
        when (powerMode) {
            ScanPowerMode.LOW_LATENCY -> builder.setReportDelay(0).setScanMode(ScanSettings.SCAN_MODE_LOW_LATENCY)
            ScanPowerMode.BALANCED -> builder.setReportDelay(1).setScanMode(ScanSettings.SCAN_MODE_BALANCED)
            ScanPowerMode.LOW_POWER -> builder.setReportDelay(1).setScanMode(ScanSettings.SCAN_MODE_LOW_POWER)
        }
    }

    @TargetApi(Build.VERSION_CODES.M)
    private fun marshmallowCallbackModeTransformer(): (ScanSettings.Builder, ScanCallbackMode) -> ScanSettings.Builder = { builder, callbackMode ->
        when (callbackMode) {
            ScanCallbackMode.FIRST_MATCH -> builder.setCallbackType(ScanSettings.CALLBACK_TYPE_FIRST_MATCH)
            ScanCallbackMode.ALL_MATCH -> builder
        }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    private fun lollipopCallbackModeTransformer(): (ScanSettings.Builder, ScanCallbackMode) -> ScanSettings.Builder = { builder, _ -> builder }

}