package com.estimote.scanning_plugin.packet_provider.parsers

import android.annotation.TargetApi
import android.os.Build
import android.os.ParcelUuid
import com.estimote.scanning_plugin.common.toHex
import com.estimote.scanning_plugin.common.toUnsignedInt
import com.estimote.scanning_plugin.packet_provider.*
import com.estimote.scanning_plugin.packet_provider.scanner.EstimoteScanResult
import com.estimote.scanning_plugin.packet_provider.use_cases.EstimoteScanResultParser
import java.nio.ByteBuffer
import java.util.*
import java.util.concurrent.TimeUnit
import kotlin.experimental.and

/**
 * @author arek.biela@estimote.com (Arek Biela).
 */

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class EstimoteTelemetryAV2Parser: EstimoteScanResultParser<EstimoteTelemetryFrameAPacket> {

    private val ESTIMOTE_SERVICE_UUID = ParcelUuid(UUID.fromString("0000fe9a-0000-1000-8000-00805f9b34fb"))

    override fun parse(result: EstimoteScanResult): EstimoteTelemetryFrameAPacket {
        val bytes = ByteBuffer.wrap(result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID))
        with (bytes) {
            return EstimoteTelemetryFrameAPacket(
                    EstimoteMacAddress(result.device.address),
                    result.rssi,
                    result.timestampNanosSinceBoot,
                    getShortIdentifier(),
                    getAcceleration(),
                    getMotionState(),
                    getCurrentMotionDuration(),
                    getPreviousMotionDuration(),
                    getGPIO(),
                    getPressure()
            )
        }
    }

    private fun ByteBuffer.getShortIdentifier(): String {
        val shortIdBytes = ByteArray(8)
        this.position(1)
        this.get(shortIdBytes, 0, 8)
        return shortIdBytes.toHex()
    }

    private fun ByteBuffer.getAcceleration(): EstimoteAcceleration {
        val x = get(10) * 2.0 / 127.0
        val y = get(11) * 2.0 / 127.0
        val z = get(12) * 2.0 / 127.0
        return EstimoteAcceleration(x, y, z)
    }

    private fun ByteBuffer.getMotionState(): Boolean {
        return get(15).and(0b00000011.toByte()).toInt() == 1
    }

    private fun getMotionDuration(byte: Byte): EstimoteMotionDuration {
        val value = byte.and(0b00111111.toByte()).toUnsignedInt()
        val unitCode = byte.and(0b11000000.toByte()).toUnsignedInt() shr 6
        val timeUnit = transformToTimeUnit(unitCode)
        val recalculatedValue = recalculateIfCountedInWeeks(value, timeUnit)
        return EstimoteMotionDuration(recalculatedValue, timeUnit)
    }
    private fun ByteBuffer.getPreviousMotionDuration() = getMotionDuration(get(13))

    private fun ByteBuffer.getCurrentMotionDuration() = getMotionDuration(get(14))

    private fun ByteBuffer.getGPIO(): EstimoteGpio {
        val pin0 = get(15).and(0b00010000.toByte()).toUnsignedInt() shr 4 == 1
        val pin1 = get(15).and(0b00100000.toByte()).toUnsignedInt() shr 5 == 1
        val pin2 = get(15).and(0b01000000.toByte()).toUnsignedInt() shr 6 == 1
        val pin3 = get(15).and(0b10000000.toByte()).toUnsignedInt() shr 7 == 1
        return EstimoteGpio(pin0, pin1, pin2, pin3)
    }

    private fun ByteBuffer.getPressure(): Double {
        val pressureBytes = ByteArray(4)
        this.position(16)
        this.get(pressureBytes, 0, 4)
        return pressureBytes.getPressureValueInPascals()
    }

    private fun ByteArray.getPressureValueInPascals(): Double {
        val reversedPressureBytes = reversedArray()
        return reversedPressureBytes.toHex().toLong(radix = 16)/ 256.0
    }

    private fun transformToTimeUnit(value: Int): TimeUnit {
        return when (value) {
            0 -> TimeUnit.SECONDS
            1 -> TimeUnit.MINUTES
            2 -> TimeUnit.HOURS
            3 -> TimeUnit.DAYS
            else -> TimeUnit.MILLISECONDS
        }
    }

    private fun recalculateIfCountedInWeeks(value: Int, timeUnit: TimeUnit): Int {
         return if (timeUnit == TimeUnit.DAYS && value > 31) 7 * value else value
    }
}