package com.estimote.scanning_plugin.packet_provider.parsers

import android.annotation.TargetApi
import android.os.Build
import android.os.ParcelUuid
import android.util.Log
import com.estimote.android_ketchup.kotlin_goodness.toHexString
import com.estimote.internal_plugins_api.scanning.EstimoteRemoteLocation.HeardPacket
import com.estimote.scanning_plugin.packet_provider.EstimoteMacAddress
import com.estimote.scanning_plugin.packet_provider.EstimoteRemoteLocationPacket
import com.estimote.scanning_plugin.packet_provider.scanner.EstimoteScanResult
import com.estimote.scanning_plugin.packet_provider.use_cases.EstimoteScanResultParser
import java.nio.ByteBuffer
import java.util.*

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class EstimoteRemoteLocationParser: EstimoteScanResultParser<EstimoteRemoteLocationPacket> {

    private val ESTIMOTE_SERVICE_UUID = ParcelUuid(UUID.fromString("0000fe9a-0000-1000-8000-00805f9b34fb"))

    override fun parse(result: EstimoteScanResult): EstimoteRemoteLocationPacket {
        val bytes = ByteBuffer.wrap(result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID))
        with(bytes) {
            return EstimoteRemoteLocationPacket(
                    deviceId(),
                    heardPackets(),
                    EstimoteMacAddress(result.device.address),
                    result.rssi,
                    result.timestampNanosSinceBoot
            )
        }
    }

    private fun ByteBuffer.deviceId(): Short {
        position(1)
        return short
    }

    private fun ByteBuffer.heardPackets(): List<HeardPacket> {
        var startPosition = 3
        val packets = mutableListOf<HeardPacket>()
        while (startPosition + 5 <= this.array().size) {
            packets += getHeardPacket(startPosition)
            startPosition += 5
        }
        return packets
    }

    private fun ByteBuffer.getHeardPacket(startPosition: Int): HeardPacket {
        val idBytes = ByteArray(4)
        position(startPosition)
        get(idBytes, 0, 4)
        return ParsedHeardPacket(idBytes.toHexString(), -get(startPosition+4).toInt())
    }

    private fun ByteBuffer.heardIdentifier(): String {
        val idBytes = ByteArray(16)
        position(1)
        get(idBytes, 0, 16)
        return idBytes.toHexString()
    }

    private fun ByteBuffer.heardRssi(): Int {
        return -get(17).toInt()
    }

    private data class ParsedHeardPacket(override val shortIdentifier: String, override val rssi: Int): HeardPacket
}