package com.estimote.scanning_plugin.api

import android.annotation.TargetApi
import android.app.Notification
import android.content.Context
import android.os.Build
import com.estimote.internal_plugins_api.scanning.BluetoothBackgroundTriggerConfigurator
import com.estimote.internal_plugins_api.scanning.BluetoothScanner
import com.estimote.scanning_plugin.dagger.Dagger
import com.estimote.scanning_plugin.packet_provider.PacketProvider
import io.reactivex.Observable
import java.lang.UnsupportedOperationException
import javax.inject.Inject


/**
 * This is main factory for creating [BluetoothScanner] objects,
 * that will allow you to get all advertised BLE packets of given supported protocol.
 *
 * @see BluetoothScanner
 * @see BluetoothScanner.ScanHandler
 */
@Suppress("unused")
class EstimoteBluetoothScannerFactory(private val applicationContext: Context) {

    @Inject
    internal lateinit var packetProvider: PacketProvider
    @Inject
    internal lateinit var bluetoothBackgroundTrigger: BluetoothBackgroundTriggerConfigurator

    private var simpleBluetoothScanner: BluetoothScanner? = null
    private var bluetoothScannerInForegroundService: BluetoothScanner? = null

    init {
        Dagger.create(applicationContext)
        Dagger.injectInTo(this)
    }

    /**
     * Creates bluetooth scanner (or returns previously created instance) and wraps it in foreground service.
     * This will allow to scan independently of your application lifecycle, but needs to be properly setup.
     *
     * *Starting and stopping scan:* To start a scan you need to call [BluetoothScanner] desired scan method,
     * and after using `start()` you will be provided with [BluetoothScanner.ScanHandler] object
     * that can be used to stop the scan. You can control the lifecycle of the scanning itself by
     * properly using `start()` and `stop()` methods in the lifecycle of your application component.
     *
     * Here are some examples:
     * 1. Scanning only during one activity being active - start your scan in activity `onStart()` method,
     * and stop it in `onStop()`
     * 2. Scanning during one full activity lifecycle until destroyed - start in activity's `onCreate()`
     * and stop in activity's `onDestroy()`
     * 3. Scanning during the lifecycle of your whole application - build scanner in your class extending android `Application`
     * 'onCreate()' method and stop it in `onDestroy()` method.
     *
     * *IMPORTANT:* you need to call `stop()` for each started scan *before* calling your activity's
     * `super.onStop()` or `super.onDestroy()` methods. This applies also to `Application` class lifecycle.
     *
     * *Using notification:* You must take care of creating it properly according to https://developer.android.com/guide/topics/ui/notifiers/notifications.html
     * Remember that in order for the notification to work on Android 8.0+ you must register Notification Channel for your app.
     * You can implement your own button in order for user to be able to stop scanning via the notification,
     * more info in the link provided above.
     *
     * *Threading:* All scanning is done and delivered on Main Thread.
     *
     * @see BluetoothScanner
     * @see BluetoothScanner.ScanHandler
     * @param notification Notification object to display in user's notification bar.
     * @return [BluetoothScanner] object wrapped in foreground service.
     */
    fun getScannerWithForegroundService(notification: Notification): BluetoothScanner =
            bluetoothScannerInForegroundService
                    ?: Dagger.initNotificationComponent(notification).getServiceWrapperHelper()
                            .let {
                                EstimoteBluetoothScanner({ it.startAndBind() }).also { bluetoothScannerInForegroundService = it}
                            }


    /**
     * Creates simple [BluetoothScanner] object (or returns previously created instance) that will handle scan synchronously with your code.
     * You can use it for single Activity purposes - like quick scan for packets around,
     * or showing list of beacons to choose.
     *
     * *Starting and stopping scan:* To start a scan you need to call [BluetoothScanner] desired scan method,
     * and after using `start()` you will be provided with [BluetoothScanner.ScanHandler] object
     * that can be used to stop the scan. You can control the lifecycle of the scanning itself by
     * properly using `start()` and `stop()` methods in the lifecycle of your application component.
     *
     * *Scan lifetime:*
     * We don't suggest using this scanner for long-running, background scanning (when your application is not in the foreground).
     * The system might stop delivering bluetooth scans and it may behave very randomly.
     * For such case consider using [EstimoteBluetoothScannerFactory.getScannerWithForegroundService] which will
     * run the scanning in foreground service. The scan will last until you, or the user, will stop it.
     *
     * *Threading:* All scanning is done and delivered on Main Thread.
     *
     * @see BluetoothScanner
     * @see BluetoothScanner.ScanHandler
     * @return [BluetoothScanner] object
     */
    fun getSimpleScanner(): BluetoothScanner =
            simpleBluetoothScanner
                    ?: EstimoteBluetoothScanner({ Observable.just(packetProvider) }).also { simpleBluetoothScanner = it }

    /**
     * Creates Bluetooth Trigger configuration - class which will allow you to configure and launch
     * action you defined when phone with your application installed detect Beacon nearby.
     * Your action (defined via pending intent) will be triggered even if your application is not running
     * while Beacon is Detected - pure magic!
     * This will allow you to wake-up/launch your application each time a phone gets close to Beacon.
     * *IMPORTANT:* This method requires android api level 26 (Oreo) or higher to work properly.
     *
     * @throws UnsupportedOperationException when called on Android with API level lower than 26 (Oreo)
     */
    @TargetApi(Build.VERSION_CODES.O)
    fun getTriggerConfigurator(): BluetoothBackgroundTriggerConfigurator =
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) bluetoothBackgroundTrigger
            else throw UnsupportedOperationException("This method is available starting from api level 26 (Oreo)")
}