package com.estimote.scanning_plugin.packet_provider.recognizers

import android.annotation.TargetApi
import android.os.Build
import android.os.ParcelUuid
import com.estimote.scanning_plugin.packet_provider.scanner.EstimoteScanResult
import com.estimote.scanning_plugin.packet_provider.use_cases.EstimoteScanResultRecognizer
import java.nio.ByteBuffer
import java.util.*
import kotlin.experimental.and

/**
 * @author Estimote Inc. (contact@estimote.com)
 */
@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class EstimoteLteRecognizer : EstimoteScanResultRecognizer {
    private val ESTIMOTE_SERVICE_UUID = ParcelUuid(UUID.fromString("0000fe9a-0000-1000-8000-00805f9b34fb"))
    private val ESTIMOTE_CONNECTIVITY_FRAME_TYPE = 0

    override fun recognize(result: EstimoteScanResult): Boolean {
        return result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID) != null
                && result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID).isNotEmpty()
                && result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID)[0].toInt() and 0x0F == ESTIMOTE_CONNECTIVITY_FRAME_TYPE
                && with (ByteBuffer.wrap(result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID))) {
            (getApplicationVersion() == "15.15.15" && getBootloaderVersion() == "15.15.15")
                    || (getApplicationVersion() == "0.0.0" && getBootloaderVersion() == "0.1.11")
                    || (getApplicationVersion() == "0.0.12" && getBootloaderVersion() == "0.0.1")
                    || (getApplicationVersion() == "0.0.13" && getBootloaderVersion() == "0.0.1")
        }
    }

    private fun ByteBuffer.getBootloaderVersion(): String {
        val bytes = ByteArray(2)
        position(18)
        get(bytes, 0, 2)
        return "${(bytes[1].toInt() shr 4).toVersionOctet()}.${bytes[1].toVersionOctet()}.${(bytes[0].toInt() shr 4).toVersionOctet()}"
    }

    private fun ByteBuffer.getApplicationVersion(): String {
        val bytes = ByteArray(2)
        position(17)
        get(bytes, 0, 2)
        return "${bytes[1].toVersionOctet()}.${(bytes[0].toInt() shr 4).toVersionOctet()}.${bytes[0].toVersionOctet()}"
    }

    private fun Byte.toVersionOctet() =
            (this and 0x0F).toString()

    private fun Int.toVersionOctet() =
            (this and 0x0F).toString()
}