package com.estimote.scanning_plugin.packet_provider.parsers

import android.annotation.TargetApi
import android.os.Build
import android.os.ParcelUuid
import com.estimote.internal_plugins_api.scanning.DeviceType
import com.estimote.scanning_plugin.common.toHex
import com.estimote.scanning_plugin.packet_provider.EstimoteLteBeaconPacket
import com.estimote.scanning_plugin.packet_provider.EstimoteMacAddress
import com.estimote.scanning_plugin.packet_provider.scanner.EstimoteScanResult
import com.estimote.scanning_plugin.packet_provider.use_cases.EstimoteScanResultParser
import java.nio.ByteBuffer
import java.util.*
import kotlin.experimental.and

/**
 * @author Estimote Inc. (contact@estimote.com)
 */
@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class EstimoteLteParser: EstimoteScanResultParser<EstimoteLteBeaconPacket> {

    private val ESTIMOTE_SERVICE_UUID = ParcelUuid(UUID.fromString("0000fe9a-0000-1000-8000-00805f9b34fb"))
    private val ESTIMOTE_MANUFACTURER_ID = 0x015D
    private val emptyManufacturerDataBytes = byteArrayOf(0x00, 0x00, 0x00, 0x00, 0x00, 0x00)

    override fun parse(result: EstimoteScanResult): EstimoteLteBeaconPacket {
        val serviceData = ByteBuffer.wrap(result.scanRecord.getServiceData(ESTIMOTE_SERVICE_UUID))
        val manufacturerSpecificData = ByteBuffer.wrap(result.scanRecord.getManufacturerSpecificData(ESTIMOTE_MANUFACTURER_ID)
                ?: emptyManufacturerDataBytes)

        with(Pair(serviceData, manufacturerSpecificData)) {
            return EstimoteLteBeaconPacket(
                    getDeviceId(),
                    getDeviceType(),
                    getBootloaderVersion(),
                    getApplicationVersion(),
                    getShakeToConnect(),
                    getNearToConnect(),
                    EstimoteMacAddress(result.device.address),
                    result.rssi,
                    result.timestampNanosSinceBoot
            )
        }
    }

    private fun Pair<ByteBuffer, ByteBuffer>.getDeviceId(): String {
        val bytes = ByteArray(16)
        first.position(1)
        first.get(bytes, 0, 16)
        return bytes.toHex()
    }

    private fun Pair<ByteBuffer, ByteBuffer>.getDeviceType(): DeviceType {
        return DeviceType.LTE_BEACON
    }

    private fun Pair<ByteBuffer, ByteBuffer>.getBootloaderVersion(): String {
        val bytes = ByteArray(2)
        first.position(18)
        first.get(bytes, 0, 2)
        return "${(bytes[1].toInt() shr 4).toVersionOctet()}.${bytes[1].toVersionOctet()}.${(bytes[0].toInt() shr 4).toVersionOctet()}"
    }

    private fun Pair<ByteBuffer, ByteBuffer>.getApplicationVersion(): String {
        val bytes = ByteArray(2)
        first.position(17)
        first.get(bytes, 0, 2)
        return "${bytes[1].toVersionOctet()}.${(bytes[0].toInt() shr 4).toVersionOctet()}.${bytes[0].toVersionOctet()}"
    }

    private fun Pair<ByteBuffer, ByteBuffer>.getShakeToConnect(): Boolean {
        return second.get(1) and 0x01 > 0
    }

    private fun Pair<ByteBuffer, ByteBuffer>.getNearToConnect(): Boolean {
        return second.get(1) and 0x02 > 0
    }

    private fun Byte.toVersionOctet() =
            (this and 0x0F).toString()

    private fun Int.toVersionOctet() =
            (this and 0x0F).toString()
}