package com.estimote.scanning_plugin.packet_provider

import android.annotation.TargetApi
import android.os.Build
import com.estimote.internal_plugins_api.scanning.*
import com.estimote.scanning_plugin.packet_provider.use_cases.*
import com.estimote.scanning_plugin.settings.*
import io.reactivex.Observable

/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class EstimotePacketProvider(private val iBeaconScanUseCase: IBeaconScanUseCase,
                                      private val eddystoneUidScanUseCase: EddystoneUidScanUseCase,
                                      private val estimoteLocationScanUseCase: EstimoteLocationScanUseCase,
                                      private val estimoteMeshScanUseCase: EstimoteMeshScanUseCase,
                                      private val estimoteConnectivityScanUseCase: EstimoteConnectivityScanUseCase,
                                      private val estimoteConnectivityIBeaconScanUseCase: EstimoteConnectivityIBeaconScanUseCase,
                                      private val estimoteNearableScanUseCase: EstimoteNearableScanUseCase,
                                      private val estimoteUwbScanUseCase: EstimoteUwbScanUseCase,
                                      private val estimoteMirrorScanUseCase: EstimoteMirrorScanUseCase,
                                      private val estimoteTelemetryAV0ScanUseCase: EstimoteTelemetryAV0ScanUseCase,
                                      private val estimoteTelemetryAV1ScanUseCase: EstimoteTelemetryAV1ScanUseCase,
                                      private val estimoteTelemetryAV2ScanUseCase: EstimoteTelemetryAV2ScanUseCase,
                                      private val estimoteTelemetryBV0ScanUseCase: EstimoteTelemetryBV0ScanUseCase,
                                      private val estimoteTelemetryBV1ScanUseCase: EstimoteTelemetryBV1ScanUseCase,
                                      private val estimoteTelemetryBV2ScanUseCase: EstimoteTelemetryBV2ScanUseCase,
                                      private val estimoteTelemetryFullScanUseCase: EstimoteTelemetryFullScanUseCase,
                                      private val estimoteSecureScanUseCase: EstimoteSecureScanUseCase,
                                      private val estimoteRescueScanUseCase: EstimoteRescueScanUseCase) : PacketProvider {

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideIBeacon(scanSettings: IBeaconScanSettings): Observable<out Beacon> {
        return iBeaconScanUseCase.run(scanSettings)
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEddystoneUid(scanSettings: EddystoneScanSettings): Observable<out EddystoneUid> {
        return eddystoneUidScanUseCase.run(scanSettings)
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEstimoteLocation(scanSettings: EstimoteLocationScanSettings): Observable<out EstimoteLocation> {
        return estimoteLocationScanUseCase.run(scanSettings).map { it as EstimoteLocation }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEstimoteMesh(scanSettings: EstimoteMeshScanSettings): Observable<out EstimoteMesh> {
        return estimoteMeshScanUseCase.run(scanSettings).map { it as EstimoteMesh }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideConnectivity(scanSettings: ConnectivityScanSettings): Observable<out EstimoteConnectivity> {
        return estimoteConnectivityScanUseCase.run(scanSettings)
                .mergeWith(estimoteConnectivityIBeaconScanUseCase.run(scanSettings))
                .map { it as EstimoteConnectivity }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideNearable(scanSettings: NearableScanSettings): Observable<out EstimoteNearable> {
        return estimoteNearableScanUseCase.run(scanSettings).map { it as EstimoteNearable }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEstimoteUwb(scanSettings: UwbScanSettings): Observable<out EstimoteUwb> {
        return estimoteUwbScanUseCase.run(scanSettings).map { it as EstimoteUwb }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEstimoteMirror(scanSettings: MirrorScanSettings): Observable<out EstimoteMirror> {
        return estimoteMirrorScanUseCase.run(scanSettings).map { it as EstimoteMirror }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEstimoteTelemetryFrameA(scanSettings: EstimoteTelemetryScanSettings): Observable<out EstimoteTelemetryFrameA> {
        return estimoteTelemetryAV0ScanUseCase.run(scanSettings)
                .mergeWith(estimoteTelemetryAV1ScanUseCase.run(scanSettings))
                .mergeWith(estimoteTelemetryAV2ScanUseCase.run(scanSettings))
                .map { it as EstimoteTelemetryFrameA }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEstimoteTelemetryFrameB(scanSettings: EstimoteTelemetryScanSettings): Observable<out EstimoteTelemetryFrameB> {
        return estimoteTelemetryBV0ScanUseCase.run(scanSettings)
                .mergeWith(estimoteTelemetryBV1ScanUseCase.run(scanSettings))
                .mergeWith(estimoteTelemetryBV2ScanUseCase.run(scanSettings))
                .map { it as EstimoteTelemetryFrameB }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEstimoteTelemetryFull(scanSettings: EstimoteTelemetryScanSettings): Observable<out EstimoteTelemetryFull> {
        return estimoteTelemetryFullScanUseCase.run(
                provideEstimoteTelemetryFrameA(scanSettings),
                provideEstimoteTelemetryFrameB(scanSettings),
                provideConnectivity(ConnectivityScanSettings(scanSettings.powerMode, scanSettings.callbackMode)))
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEstimoteSecure(scanSettings: EstimoteSecureScanSettings): Observable<out EstimoteSecure> {
        return estimoteSecureScanUseCase.run(scanSettings).map { it as EstimoteSecure }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    override fun provideEstimoteRescue(scanSettings: EstimoteRescueScanSettings): Observable<out EstimoteRescue> {
        return estimoteRescueScanUseCase.run(scanSettings).map { it as EstimoteRescue }
    }
}