package com.estimote.scanning_plugin.api.use_cases

import com.estimote.scanning_plugin.packet_provider.PacketProvider
import io.reactivex.Observable
import io.reactivex.disposables.Disposable
import java.util.concurrent.TimeUnit

internal class ScanUseCase<in SETTINGS_TYPE, out PACKET_TYPE>(private val loadPacketProvider: () -> Observable<PacketProvider>, private val startScan: (PacketProvider, SETTINGS_TYPE) -> Observable<out PACKET_TYPE>) {

    fun run(timeoutValue: Long, timeUnit: TimeUnit, settings: SETTINGS_TYPE, onPacketFound: (PACKET_TYPE) -> Unit, onErrorOccurred: (Throwable) -> Unit) =
            whenPacketProviderGetsReady()
                    .scanForPacketsAccordingToSettings(settings)
                    .applyUserTimeout(timeoutValue, timeUnit)
                    .notifyUserCallbacks(onPacketFound, onErrorOccurred)
                    .createHandlerForThisScanRequest()


    private fun whenPacketProviderGetsReady() =
            loadPacketProvider()

    private fun Observable<PacketProvider>.scanForPacketsAccordingToSettings(settings: SETTINGS_TYPE) =
            this.flatMap { startScan(it, settings) }

    private fun Observable<PACKET_TYPE>.applyUserTimeout(timeoutValue: Long, timeUnit: TimeUnit): Observable<PACKET_TYPE> =
            if (timeoutValue == 0L) this else this.timeout(timeoutValue, timeUnit)

    private fun Observable<PACKET_TYPE>.notifyUserCallbacks(onPacketFound: (PACKET_TYPE) -> Unit, onErrorOccurred: (Throwable) -> Unit) =
            this.subscribe({ onPacketFound(it) }, { onErrorOccurred(it) })

    private fun Disposable.createHandlerForThisScanRequest() =
            this.let { RxScanHandler(it) }
}