package com.estimote.scanning_plugin.packet_provider.scanner

import android.annotation.TargetApi
import android.bluetooth.le.*
import android.os.Build
import com.estimote.scanning_plugin.packet_provider.use_cases.EstimoteScanner
import com.estimote.scanning_plugin.settings.EstimoteScanSettings
import com.estimote.scanning_plugin.settings.ScanSettingsVisitor
import com.wafel.skald.api.createLogger
import io.reactivex.Observable

/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class PostLollipopEstimoteScanner(private val bluetoothLeScanner: BluetoothLeScanner,
                                           private val scanSettingsTransformer: ScanSettingsVisitor<ScanSettings>,
                                           private val scanFiltersTransformer: ScanSettingsVisitor<List<ScanFilter>>) : EstimoteScanner {

    private val logger = createLogger(this.javaClass)

    @TargetApi(Build.VERSION_CODES.N)
    override fun scan(scanSettings: EstimoteScanSettings): Observable<EstimoteScanResult> {
        return Observable.create { emitter ->
            val callback = object : ScanCallback() {
                override fun onScanResult(callbackType: Int, result: ScanResult) = emitter.onNext(result.toEstimoteScanResult())
                override fun onBatchScanResults(results: MutableList<ScanResult>) = results.forEach { emitter.onNext(it.toEstimoteScanResult()) }
                override fun onScanFailed(errorCode: Int) {
                    if (!emitter.isDisposed) emitter.onError(Exception("Bluetooth Low Energy scan failed with error code: $errorCode"))
                }
            }
            emitter.setCancellable {
                catchBluetoothErrors("Unable to stop scan while Bluetooth Adapter is not turned ON (Bluetooth is probably disabled)", {
                    bluetoothLeScanner.stopScan(callback)
                })
            }
            catchBluetoothErrors("Unable to start scan while Bluetooth Adapter is not turned ON (Bluetooth is probably disabled)", {
                bluetoothLeScanner.startScan(scanSettings.accept(scanFiltersTransformer), scanSettings.accept(scanSettingsTransformer), callback)
            }, { emitter.onError(it) })
        }
    }

    private fun ScanResult.toEstimoteScanResult() = EstimoteScanResult(device, rssi, timestampNanos, scanRecord)

    private inline fun catchBluetoothErrors(message: String, action: () -> Unit, errorAction: (Throwable) -> Unit = {}) {
        try {
            action()
        } catch (e: IllegalStateException) {
            logger.warn(message)
            errorAction(IllegalStateException(message))
        }
    }
}