package com.estimote.scanning_plugin.packet_provider.rssi_smoothers

import com.estimote.scanning_plugin.packet_provider.use_cases.ExpirationCache

/**
 * Non-thread safe cache implementation that expires its items after given period of time.
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class TimeExpirationCache<in KEY, VALUE>(private val timer: Timer,
                                         private val loader: (KEY) -> VALUE,
                                         private val expirationPolicy: ExpirationPolicy) : ExpirationCache<KEY, VALUE> {

    private val map = mutableMapOf<KEY, Pair<Long, VALUE>>()

    override fun get(key: KEY): VALUE {
        removeExpiredValues()
        val value = map[key]
        return if (value == null || expirationPolicy.hasExpired(value.first)) {
            loadNewValueForKey(key)
        } else {
            updateTimestampForKey(key, value.second)
        }
    }

    private fun loadNewValueForKey(key: KEY) : VALUE {
        val newValue = loader.invoke(key)
        put(key, newValue)
        return newValue
    }

    private fun updateTimestampForKey(key: KEY, value: VALUE) : VALUE{
        put(key, value)
        return value
    }

    override fun put(key: KEY, value: VALUE) {
        map.put(key, (timer.getCurrentTimeMillis() to value))
    }

    override fun removeExpiredValues() {
        val iterator = map.iterator()
        while (iterator.hasNext()) {
            val entry = iterator.next()
            if (expirationPolicy.hasExpired(entry.value.first)) iterator.remove()
        }
    }

    override fun contains(key: KEY): Boolean = map.contains(key)


}