package com.estimote.repository_plugin

import io.reactivex.Completable
import io.reactivex.Maybe
import io.reactivex.Single

/**
 * @author Estimote Inc. (contact@estimote.com)
 */
interface Repository<in KEY, DATA_MODEL> {
    fun get(key: KEY): Single<DATA_MODEL>
}

fun <KEY, DATA_MODEL> buildRepository(init: RepositoryBuilder<KEY, DATA_MODEL>.() -> Unit): Repository<KEY, DATA_MODEL> {
    val builder = SimpleRepositoryBuilder<KEY, DATA_MODEL>().apply(init)
    val upstream = builder.getUpstream()
    val downstream = builder.getDownstream()
    return DefaultRepository(upstream.getFetcher(), upstream.getHasNewDataChecker(), downstream.getReader(), downstream.getWriter())
}

abstract class RepositoryBuilder<KEY, DATA_MODEL> {
    abstract fun withUpstream(init: UpstreamBuilder<KEY, DATA_MODEL>.() -> Unit): RepositoryBuilder<KEY, DATA_MODEL>
    abstract fun withDownstream(init: DownstreamBuilder<KEY, DATA_MODEL>.() -> Unit): RepositoryBuilder<KEY, DATA_MODEL>
    internal abstract fun getUpstream(): UpstreamBuilder<KEY, DATA_MODEL>
    internal abstract fun getDownstream(): DownstreamBuilder<KEY, DATA_MODEL>
}

abstract class DownstreamBuilder<KEY, DATA_MODEL> {
    abstract fun reader(reader: (KEY) -> Maybe<DATA_MODEL>)
    abstract fun writer(writer: (KEY, DATA_MODEL) -> Completable)
    internal abstract fun getReader(): (KEY) -> Maybe<DATA_MODEL>
    internal abstract fun getWriter(): (KEY, DATA_MODEL) -> Completable
}

abstract class UpstreamBuilder<KEY, DATA_MODEL> {
    abstract fun fetcher(fetcher: (KEY) -> Single<DATA_MODEL>)
    abstract fun hasNewDataChecker(hasNewDataChecker: (KEY) -> Single<Boolean>)
    internal abstract fun getFetcher(): (KEY) -> Single<DATA_MODEL>
    internal abstract fun getHasNewDataChecker(): (KEY) -> Single<Boolean>
}

class SimpleRepositoryBuilder<KEY, DATA_MODEL> : RepositoryBuilder<KEY, DATA_MODEL>() {

    private var upstreamBuilder: UpstreamBuilder<KEY, DATA_MODEL> = SimpleUpstreamBuilder()
    private var downstreamBuilder: DownstreamBuilder<KEY, DATA_MODEL> = SimpleDownstreamBuilder()
    override fun withUpstream(init: UpstreamBuilder<KEY, DATA_MODEL>.() -> Unit): RepositoryBuilder<KEY, DATA_MODEL> {
        SimpleUpstreamBuilder<KEY, DATA_MODEL>().apply(init).also { upstreamBuilder = it }
        return this
    }
    override fun withDownstream(init: DownstreamBuilder<KEY, DATA_MODEL>.() -> Unit): RepositoryBuilder<KEY, DATA_MODEL> {
        SimpleDownstreamBuilder<KEY, DATA_MODEL>().apply(init).also { downstreamBuilder = it }
        return this
    }
    override fun getUpstream() = upstreamBuilder
    override fun getDownstream() = downstreamBuilder

}

class SimpleUpstreamBuilder<KEY, DATA_MODEL> : UpstreamBuilder<KEY, DATA_MODEL>() {
    private var fetcher: (KEY) -> Single<DATA_MODEL> = { Single.error(Throwable("Unable to fetch data - upstream not configured.")) }
    private var hasNewDataChecker: (KEY) -> Single<Boolean> = { Single.just(false) }
    override fun fetcher(fetcher: (KEY) -> Single<DATA_MODEL>) { this.fetcher = fetcher }
    override fun hasNewDataChecker(hasNewDataChecker: (KEY) -> Single<Boolean>) { this.hasNewDataChecker = hasNewDataChecker }
    override fun getFetcher() = fetcher
    override fun getHasNewDataChecker() = hasNewDataChecker
}

class SimpleDownstreamBuilder<KEY, DATA_MODEL> : DownstreamBuilder<KEY, DATA_MODEL>() {
    private var reader: (KEY) -> Maybe<DATA_MODEL> = { Maybe.empty() }
    private var writer: (KEY, DATA_MODEL) -> Completable = { _, _ -> Completable.complete() }
    override fun reader(reader: (KEY) -> Maybe<DATA_MODEL>) { this.reader = reader }
    override fun writer(writer: (KEY, DATA_MODEL) -> Completable) { this.writer = writer }
    override fun getReader() = reader
    override fun getWriter() = writer

}
