package com.estimote.repository_plugin

import com.wafel.skald.api.createLogger
import io.reactivex.Completable
import io.reactivex.Maybe
import io.reactivex.Single

/**
 * @author Estimote Inc. (contact@estimote.com)
 */
class DefaultRepository<in KEY, DATA_MODEL>(
        private val upstreamFetch: (KEY) -> Single<DATA_MODEL>,
        private val upstreamHasNewData: (KEY) -> Single<Boolean>,
        private val downstreamRead: (KEY) -> Maybe<DATA_MODEL>,
        private val downstreamWrite: (KEY, DATA_MODEL) -> Completable
) : Repository<KEY, DATA_MODEL> {

    private val logger = createLogger(DefaultRepository::class.java)

    override fun get(key: KEY): Single<DATA_MODEL> =
            checkIfUpstreamHasNewData(key)
                    .onErrorReturn { false }
                    .flatMap { hasNewData ->
                        if (hasNewData)
                            fetchDataFromUpstream(key)
                                    .thenWriteItToDownstream(key)
                                    .onErrorReadDataFromDownstream(key)
                        else
                            readDataFromDownstream(key)
                                    .onErrorFetchDataFromUpstream(key)
                    }

    private fun readDataFromDownstream(key: KEY): Single<DATA_MODEL> {
        return downstreamRead(key)
                .toSingle()
                .doOnError { logger.debug("Unable to read data from the downstream.") }
    }

    private fun checkIfUpstreamHasNewData(key: KEY): Single<Boolean> {
        return upstreamHasNewData(key)
                .doOnError { logger.debug("Unable to check if upstream has a new data.") }
    }

    private fun fetchDataFromUpstream(key: KEY): Single<DATA_MODEL> {
        return upstreamFetch(key)
                .doOnError { logger.debug("Unable to fetch data from the upstream.") }
    }

    private fun Single<DATA_MODEL>.thenWriteItToDownstream(key: KEY) =
            flatMap {
                downstreamWrite(key, it)
                        .doOnError { logger.debug("Unable to write data to the downstream.") }
                        .onErrorResumeNext { Completable.complete() }
                        .andThen(Single.just(it))
            }


    private fun Single<DATA_MODEL>.onErrorReadDataFromDownstream(key: KEY) =
            onErrorResumeNext { readDataFromDownstream(key) }

    private fun Single<DATA_MODEL>.onErrorFetchDataFromUpstream(key: KEY) =
            onErrorResumeNext { fetchDataFromUpstream(key).thenWriteItToDownstream(key) }
}