package com.estimote.managemet_sdk.configuration_manager.use_case

import com.estimote.managemet_sdk.configuration_manager.cloud.CloudCommands
import com.estimote.managemet_sdk.configuration_manager.cloud.CloudDeviceId
import com.estimote.managemet_sdk.configuration_manager.cloud.ManagementCloud
import com.estimote.managemet_sdk.configuration_manager.cloud.LegacySettingsManager

import com.estimote.managemet_sdk.configuration_manager.internals.SyncableDeviceVisitorUseCase
import com.estimote.managemet_sdk.configuration_manager.internals.VisitableBeacon
import io.reactivex.Completable
import io.reactivex.Observable

internal class SyncCommandsToTheDeviceUseCase(private val managementCloud: ManagementCloud,
                                              private val legacySettingsManager: LegacySettingsManager) : SyncableDeviceVisitorUseCase<Completable>() {
    override fun useCaseMinFirmwareVersion() = "4.13.0"

    override fun onFirmwareVersionNotSupported() = Completable.complete()

    override fun onFirmwareVersionSupported(device: VisitableBeacon): Completable {
        return whenCommandsAreObtained(device)
                .thenSaveEachOfThemInToDevice()
                .andConfirmThemForThisDevice(device)
    }

    private fun whenCommandsAreObtained(device: VisitableBeacon): Observable<String> {
        return managementCloud.getCommandsForDevice(CloudDeviceId(device.configurableDevice.deviceId.toHexString())).map { it.commands }.flatMapIterable {it}
    }

    private fun Observable<String>.thenSaveEachOfThemInToDevice(): Observable<String> {
        return this.flatMapSingle { command -> legacySettingsManager.saveCommand(command).toSingle { command } }
    }

    private fun Observable<String>.andConfirmThemForThisDevice(device: VisitableBeacon): Completable {
        return this.flatMapCompletable { command -> managementCloud.confirmCommands(CloudDeviceId(device.configurableDevice.deviceId.toHexString()), CloudCommands(listOf(command))) }
    }
}
