package com.estimote.managemet_sdk.configuration_manager.use_case


import com.estimote.managemet_sdk.configuration_manager.cloud.CloudDeviceId
import com.estimote.managemet_sdk.configuration_manager.cloud.CloudMeshMessage
import com.estimote.managemet_sdk.configuration_manager.cloud.ManagementCloud
import com.estimote.managemet_sdk.configuration_manager.cloud.LegacySettingsManager
import com.estimote.managemet_sdk.configuration_manager.internals.SyncableDeviceVisitorUseCase
import com.estimote.managemet_sdk.configuration_manager.internals.VisitableBeacon
import io.reactivex.Completable
import io.reactivex.Observable


internal class SyncBeaconMeshMessageUseCase(private val managementCloud: ManagementCloud,
                                            private val legacySettingsManager: LegacySettingsManager) : SyncableDeviceVisitorUseCase<Completable>() {
    override fun useCaseMinFirmwareVersion(): String =
            "4.13.0"

    override fun onFirmwareVersionNotSupported(): Completable =
            Completable.complete()

    override fun onFirmwareVersionSupported(device: VisitableBeacon): Completable =
            whenCloudMeshMessageIsObtained(device)
                    .saveItToTheDevice()
                    .postUpdatedSettingsVersionToCloud(device)
                    .andConfirmCommandsArePushedToMesh()

    private fun whenCloudMeshMessageIsObtained(device: VisitableBeacon): Observable<CloudMeshMessage> =
            managementCloud.getMeshMessageForDevice(CloudDeviceId(device.configurableDevice.deviceId.toHexString()))

    private fun Observable<CloudMeshMessage>.saveItToTheDevice(): Observable<CloudMeshMessage> =
            this.flatMapSingle { message -> legacySettingsManager.saveMeshMessageToDevice(message) }

    private fun Observable<CloudMeshMessage>.postUpdatedSettingsVersionToCloud(device: VisitableBeacon): Observable<CloudMeshMessage> =
            this.flatMap { message -> managementCloud.updateSettingsVersion(device.configurableDevice.deviceId.toHexString(), message) }

    private fun Observable<CloudMeshMessage>.andConfirmCommandsArePushedToMesh() =
            this.flatMapCompletable { managementCloud.putMeshMessageBuffer(it) }

    private fun Observable<ManagementCloud.RequestStatus>.finishUseCase() =
            this.flatMapCompletable { Completable.complete() }

}

