package com.estimote.managemet_sdk.configuration_manager.use_case

import com.estimote.internal_plugins_api.cloud.management.CloudCommands
import com.estimote.internal_plugins_api.cloud.management.CloudDeviceId
import com.estimote.internal_plugins_api.cloud.management.ManagementCloud
import com.estimote.internal_plugins_api.legacy_settings.LegacySettingsManager
import com.estimote.managemet_sdk.configuration_manager.internals.SyncableDeviceVisitorUseCase
import com.estimote.managemet_sdk.configuration_manager.internals.VisitableBeacon
import io.reactivex.*

internal class SyncCommandsToTheDeviceUseCase(private val managementCloud: ManagementCloud, private val legacySettingsManager: LegacySettingsManager) : SyncableDeviceVisitorUseCase<Completable>() {
    override fun useCaseMinFirmwareVersion() = "4.13.0"

    override fun onFirmwareVersionNotSupported() = Completable.complete()

    override fun onFirmwareVersionSupported(device: VisitableBeacon): Completable {
        return whenCommandsAreObtained(device)
                .thenSaveEachOfThemInToDevice()
                .andConfirmThemForThisDevice(device)
    }

    private fun whenCommandsAreObtained(device: VisitableBeacon): Observable<String> {
        return managementCloud.getCommandsForDevice(CloudDeviceId(device.configurableDevice.deviceId.toHexString())).map { it.commands }.flatMapIterable {it}
    }

    private fun Observable<String>.thenSaveEachOfThemInToDevice(): Observable<String> {
        return this.flatMapSingle { command -> legacySettingsManager.saveCommand(command).toSingle { command } }
    }

    private fun Observable<String>.andConfirmThemForThisDevice(device: VisitableBeacon): Completable {
        return this.flatMapCompletable { command -> managementCloud.confirmCommands(CloudDeviceId(device.configurableDevice.deviceId.toHexString()), CloudCommands(listOf(command))) }
    }
}
