package com.estimote.managemet_sdk.use_case

import com.estimote.internal_plugins_api.cloud.CloudDeviceId
import com.estimote.internal_plugins_api.cloud.CloudManager
import com.estimote.internal_plugins_api.legacy_settings.LegacySettingsManager
import com.estimote.managemet_sdk.internals.SyncableDeviceVisitorUseCase
import com.estimote.managemet_sdk.internals.VisitableBeacon
import io.reactivex.*

internal class SyncCommandsToTheDeviceUseCase(private val cloudManager: CloudManager, private val legacySettingsManager: LegacySettingsManager) : SyncableDeviceVisitorUseCase<Completable>() {
    override fun useCaseMinFirmwareVersion() = "4.13.0"

    override fun onFirmwareVersionNotSupported() = Completable.complete()

    override fun onFirmwareVersionSupported(device: VisitableBeacon): Completable {
        return whenCommandsAreObtained(device)
                .thenSaveEachOfThemInToDevice()
                .andConfirmThemForThisDevice(device)
    }

    private fun whenCommandsAreObtained(device: VisitableBeacon): Observable<String> {
        return cloudManager.getCommandsForDevice(CloudDeviceId(device.configurableDevice.deviceId.toHexString())).map { it.commands }.flatMapIterable {it}
    }

    private fun Observable<String>.thenSaveEachOfThemInToDevice(): Observable<String> {
        return this.flatMapSingle { command -> legacySettingsManager.saveCommand(command).toSingle { command } }
    }

    private fun Observable<String>.andConfirmThemForThisDevice(device: VisitableBeacon): Completable {
        return this.flatMapCompletable { _ -> cloudManager.deleteCommands(CloudDeviceId(device.configurableDevice.deviceId.toHexString())) }
    }
}
