package com.estimote.managemet_sdk.use_case


import com.estimote.internal_plugins_api.cloud.CloudDeviceId
import com.estimote.internal_plugins_api.cloud.CloudManager
import com.estimote.internal_plugins_api.cloud.CloudManager.RequestStatus
import com.estimote.internal_plugins_api.cloud.CloudMeshMessage
import com.estimote.internal_plugins_api.legacy_settings.LegacySettingsManager
import com.estimote.managemet_sdk.internals.SyncableDeviceVisitorUseCase
import com.estimote.managemet_sdk.internals.VisitableBeacon
import io.reactivex.Completable
import io.reactivex.Observable


internal class SyncBeaconMeshMessageUseCase(private val cloudManager: CloudManager, private val legacySettingsManager: LegacySettingsManager) : SyncableDeviceVisitorUseCase<Completable>() {
    override fun useCaseMinFirmwareVersion(): String =
            "4.13.0"

    override fun onFirmwareVersionNotSupported(): Completable =
            Completable.complete()

    override fun onFirmwareVersionSupported(device: VisitableBeacon): Completable =
            whenCloudMeshMessageIsObtained(device)
                    .saveItToTheDevice()
                    .postUpdatedSettingsVersionToCloud(device)
                    .andConfirmCommandsArePushedToMesh()

    private fun whenCloudMeshMessageIsObtained(device: VisitableBeacon): Observable<CloudMeshMessage> =
            cloudManager.getMeshMessageForDevice(CloudDeviceId(device.configurableDevice.deviceId.toHexString()))

    private fun Observable<CloudMeshMessage>.saveItToTheDevice(): Observable<CloudMeshMessage> =
            this.flatMapSingle { message -> legacySettingsManager.saveMeshMessageToDevice(message) }

    private fun Observable<CloudMeshMessage>.postUpdatedSettingsVersionToCloud(device: VisitableBeacon): Observable<CloudMeshMessage> =
            this.flatMap { message -> cloudManager.updateSettingsVersion(device.configurableDevice.deviceId.toHexString(), message) }

    private fun Observable<CloudMeshMessage>.andConfirmCommandsArePushedToMesh() =
            this.flatMapCompletable { cloudManager.putMeshMessageBuffer(it) }

    private fun Observable<RequestStatus>.finishUseCase() =
            this.flatMapCompletable { Completable.complete() }

}

