package com.estimote.indoorsdk_module.algorithm.position

import com.estimote.indoorsdk_module.algorithm.model.BeaconWithDistance
import com.estimote.indoorsdk_module.cloud.LocationBeacon
import com.estimote.indoorsdk_module.cloud.LocationPosition

/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class MeanPositionCalculator(beaconsWithVoronoiCenters: Map<LocationBeacon, LocationPosition>) : PositionCalculator {

    val beaconWithMeanToVornoiCenter: Map<LocationBeacon, LocationPosition>

    init {
        beaconWithMeanToVornoiCenter = beaconsWithVoronoiCenters.mapValues {
            val meanX = (it.key.position.x + it.value.x) / 2
            val meanY = (it.key.position.y + it.value.y) / 2
            return@mapValues LocationPosition(meanX, meanY)
        }
    }

    override fun calculatePosition(beacons: List<BeaconWithDistance>): LocationPosition {
        when (beacons.size) {
            0 -> throw IllegalArgumentException()
            1 -> return voronoiPositionFromBeacon(beacons[0])
            else -> return meanPositionFromSeveralBeacons(beacons)
        }
    }

    private fun voronoiPositionFromBeacon(beacon: BeaconWithDistance): LocationPosition {
        return beaconWithMeanToVornoiCenter[beacon.beacon]!!
    }


    private fun meanPositionFromSeveralBeacons(beacons: List<BeaconWithDistance>): LocationPosition {
        val weightSum = beacons.fold(0.0, { sum, beacon -> sum + (1 / Math.pow(beacon.distance, 2.0)) })
        val x = beacons.fold(0.0, { sum, beacon -> sum + beacon.beacon.position.x * (1 / Math.pow(beacon.distance, 2.0)) }) / weightSum
        val y = beacons.fold(0.0, { sum, beacon -> sum + beacon.beacon.position.y * (1 / Math.pow(beacon.distance, 2.0)) }) / weightSum
        return LocationPosition(x, y, 0.0)
    }

}