package com.estimote.indoorsdk_module.algorithm.geometry



/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class PlanarGeometry {

    fun calculateCentroidOfPolygon(polygon: Polygon): Point2D {
        var x = 0.0
        var y = 0.0
        var area = 0.0

        for (i in 0..(polygon.size - 1)) {
            val point1 = polygon[i]
            val point2 = polygon[(i + 1) % polygon.size]

            val temp = point1.x * point2.y - point2.x * point1.y
            x += (point1.x + point2.x) * temp
            y += (point1.y + point2.y) * temp
            area += temp
        }

        area /= 2.0

        x /= 6.0 * area
        y /= 6.0 * area

        return Point2D(x, y)
    }

    fun intersectionOfPolygons(polygonA: Polygon, polygonB: Polygon, epsilon: Double): Polygon {
        val pointsFromAInsideB = polygonA.filter {
            isPointInsidePolygon(it, polygonB)
        }
        val pointsFromBInsideA = polygonB.filter {
            isPointInsidePolygon(it, polygonA)
        }
        val intersections = polygonA.getWalls().flatMap { wallA ->
            polygonB.getWalls().map { wallB ->
                try {
                    return@map intersectionsOfLines(wallA, wallB, epsilon)
                } catch (e: IllegalArgumentException) {
                    return@map null
                }
            }
        }.filterNotNull()
        val cellBoundaryPoints =  removeDuplicatedPoints(pointsFromAInsideB + pointsFromBInsideA + intersections, epsilon)
        return calculateConvexHullOfPolygon(cellBoundaryPoints)
    }

    fun removeDuplicatedPoints(polygon: Polygon, epsilon: Double): Polygon {
        if (polygon.isEmpty() || polygon.size == 1) return polygon
        val polygonWithoutDuplicates = mutableListOf<Point2D>()
        polygon.forEach { polygonPoint ->
            val isDuplicated = polygonWithoutDuplicates.filter { distanceBetweenPoints(polygonPoint, it) < epsilon }.isNotEmpty()
            if (!isDuplicated) polygonWithoutDuplicates.add(polygonPoint)
        }
        return polygonWithoutDuplicates.toList()
    }

    fun intersectionsOfLines(lineA: Line, lineB: Line, epsilon: Double): Point2D {
        val intersection = intersectionOfLinesInInfinity(lineA, lineB, epsilon)
        if ((pointToLineDistance(intersection, lineA) > epsilon) || (pointToLineDistance(intersection, lineB) > epsilon)) {
            throw IllegalArgumentException("The lines are not intersecting.")
        }
        return intersection
    }

    private fun pointToLineDistance(point: Point2D, line: Line): Double {
        return Math.abs(
                distanceBetweenPoints(line.first, point) + distanceBetweenPoints(line.second, point)
                        - distanceBetweenPoints(line.first, line.second))
    }

    fun intersectionOfLinesInInfinity(lineA: Line, lineB: Line, epsilon: Double): Point2D {
        val a1 = lineA.first
        val a2 = lineA.second
        val b1 = lineB.first
        val b2 = lineB.second
        val denominator = (a1.x - a2.x) * (b1.y - b2.y) - (a1.y - a2.y) * (b1.x - b2.x)
        if (Math.abs(denominator) < epsilon || denominator.isNaN()) throw IllegalArgumentException("Unable to find intersection of lines with denominator = 0")
        val xNominator = (a1.x * a2.y - a1.y * a2.x) * (b1.x - b2.x) - (a1.x - a2.x) * (b1.x * b2.y - b1.y * b2.x)
        val yNominator = (a1.x * a2.y - a1.y * a2.x) * (b1.y - b2.y) - (a1.y - a2.y) * (b1.x * b2.y - b1.y * b2.x)
        return Point2D(xNominator / denominator, yNominator / denominator)
    }

    /**
     * Algorithm based on http://alienryderflex.com/polygon/ .
     */
    fun isPointInsidePolygon(point: Point2D, polygon: Polygon): Boolean {
        val x = point.x
        val y = point.y
        var isInside = false
        var j = polygon.size - 1
        polygon.withIndex().forEach { (index, _) ->
            val pointI = polygon[index]
            val pointJ = polygon[j]
            if (((pointI.y < y && pointJ.y >= y) || (pointJ.y < y && pointI.y >= y)) && (pointI.x <= x || pointJ.x <= x)) {
                isInside = isInside.xor((pointI.x + (y - pointI.y) / (pointJ.y - pointI.y) * (pointJ.x - pointI.x) < x))
            }
            j = index
        }
        return isInside
    }

    fun distanceBetweenPoints(pointA: Point2D, pointB: Point2D): Double {
        return Math.sqrt(Math.pow(pointA.x - pointB.x, 2.0) + Math.pow(pointA.y - pointB.y, 2.0))
    }

    /**
     * Calculates containing triangle for two points.
     * Point MAX must be > point MIN
     */
    fun calculateTriangleContainingPoints(min: Point2D, max: Point2D, threshold: Int) : Triangle2D {
        val hypotenuseThresholdCorrection = 7
        val A = Point2D(min.x - threshold, min.y - threshold)
        val B = Point2D(min.x - threshold, threshold + hypotenuseThresholdCorrection + max.y + max.x - min.x)
        val C = Point2D(threshold + hypotenuseThresholdCorrection + max.y - min.y + max.x, min.y - threshold)
        return Triangle2D(A, B, C)
    }

    fun calculateConvexHullOfPolygon(polygon: Polygon): Polygon {
        if (polygon.isEmpty()) return emptyList()
        if (polygon.size <= 1) return polygon
        val cross = { A: Point2D, B: Point2D, C: Point2D ->
            ((A.x - C.x) * (B.y - C.y) - (A.y - C.y) * (B.x - C.x)).toLong()
        }
        val sortedPoints = polygon.sorted()
        val n = polygon.size
        var k = 0
        val convexHull = arrayOfNulls<Point2D>(n * 2)

        // Build lower hull
        for (i in 0..n - 1) {
            while (k >= 2 && cross(convexHull[k - 2]!!, convexHull[k - 1]!!, sortedPoints[i]) <= 0) k--
            convexHull[k++] = sortedPoints[i]
        }

        // Build upper hull
        var i = n - 2
        val t = k + 1
        while (i >= 0) {
            while (k >= t && cross(convexHull[k - 2]!!, convexHull[k - 1]!!, sortedPoints[i]) <= 0) k--
            convexHull[k++] = sortedPoints[i]
            i--
        }
        if (k > 1) return convexHull.toList().filterNotNull().take(k - 1) else return convexHull.toList().filterNotNull()
    }

    fun Polygon.getWalls(): List<Line> {
        return mapIndexed { index, point -> (point to this[(index + 1) % this.size]) }
    }

}