package com.estimote.indoorsdk_module.algorithm

import com.estimote.android_ketchup.rx_goodness.rollingBuffer
import com.estimote.indoorsdk_module.algorithm.distance.RssiToDistanceCalculator
import com.estimote.indoorsdk_module.algorithm.filter.DistanceFilter
import com.estimote.indoorsdk_module.algorithm.filter.HumanWalkStatefulDistanceFilter
import com.estimote.indoorsdk_module.algorithm.filter.ScannedBeaconToLocationBeaconMapper
import com.estimote.indoorsdk_module.algorithm.model.BeaconWithDistance
import com.estimote.indoorsdk_module.algorithm.model.BeaconWithRssi
import com.estimote.indoorsdk_module.algorithm.model.ScannedBeacon
import com.estimote.indoorsdk_module.algorithm.position.ExponentialPositionFilter
import com.estimote.indoorsdk_module.algorithm.position.JumpRejectionFilter
import com.estimote.indoorsdk_module.algorithm.position.PositionCalculator
import com.estimote.indoorsdk_module.algorithm.sensor.MotionDetector
import com.estimote.indoorsdk_module.algorithm.sensor.StepMotionDetector
import com.estimote.indoorsdk_module.algorithm.sensor.StepMotionDetectorSubscriber
import com.estimote.indoorsdk_module.analytics.AnalyticsEventType
import com.estimote.indoorsdk_module.analytics.AnalyticsManager
import com.estimote.indoorsdk_module.cloud.Location
import com.estimote.indoorsdk_module.cloud.LocationPosition
import com.estimote.indoorsdk_module.common.config.ConfigFactory
import com.estimote.indoorsdk_module.common.state.LocationState
import com.estimote.indoorsdk_module.common.state.LocationStateChange
import com.estimote.indoorsdk_module.common.threading.SchedulerProvider
import com.estimote.indoorsdk_module.logging.IndoorLogger
import com.estimote.internal_plugins_api.scanning.Beacon
import com.estimote.internal_plugins_api.scanning.EstimoteLocation
import com.wafel.skald.api.createLogger
import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import io.reactivex.Observable
import io.reactivex.disposables.Disposable
import io.reactivex.subjects.PublishSubject
import java.util.concurrent.TimeUnit
import javax.inject.Inject


/**
 * This is the manager which is responsible of all the positioning algorithms.
 * You need to provide it with scanned beacon data using method onScannedPackets
 *
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class EstimoteIndoorLocationManager @Inject constructor(private val mLocation: Location,
                                                                 private val mPositionUpdateIntervalMillis: Long,
                                                                 private val motionDetectorDataSubscriber: StepMotionDetectorSubscriber,
                                                                 private val mScannedBeaconToLocationBeaconMapper: ScannedBeaconToLocationBeaconMapper,
                                                                 private val mRssiToDistanceCalculator: RssiToDistanceCalculator,
                                                                 private val mHumanWalkStatefulDistanceFilter: HumanWalkStatefulDistanceFilter,
                                                                 private val mBeaconDistanceFilter: DistanceFilter,
                                                                 private val mPositionCalculator: PositionCalculator,
                                                                 private val mJumpRejectionFilter: JumpRejectionFilter,
                                                                 private val mPositionSmoothingFilter: ExponentialPositionFilter,
                                                                 private val mAnalyticsManager: AnalyticsManager,
                                                                 private val mSchedulers: SchedulerProvider,
                                                                 private val configFactory: ConfigFactory,
                                                                 private val logger: IndoorLogger,
                                                                 private val stepMotionDetector: MotionDetector) : IndoorLocationManager {

    private val skald = createLogger(this::class.java)
    private var mOnPositionUpdateListener: OnPositionUpdateListener? = null
    private var packetsQueue = PublishSubject.create<ScannedBeacon>()
    private var positioningHandle: Disposable? = null
    private var outsideHandlingHandle: Disposable? = null


    /**
     * Notifies Manager to start it's positioning algorithms. Make sure to call this method before starting scanning.
     */
    override fun startPositioning() {
        mAnalyticsManager.generateEvent(mLocation.identifier, AnalyticsEventType.START_POSITIONING)
        motionDetectorDataSubscriber.subscribe()
        stepMotionDetector.start()

        val sharedPacketQueue = packetsQueue
                .observeOn(mSchedulers.computation())
                .doOnNext{ skald.debug("packetQueue - attempting to process packet: $it") }
                .filterOutBeaconsFromOutsideOfLocation()
                .doOnNext{ skald.debug("packetQueue - packet in location: $it") }
                .keepBeaconPacketsBufferedByConfiguredTime()
                .doOnNext{ skald.debug("packetQueue - buffered packets: $it") }
                .toFlowable(BackpressureStrategy.LATEST)
                .mapToBeaconsWithRssiInLocation()
                .doOnNext{ skald.debug("packetQueue - Beacons with Rssi: $it") }
                .calculateDistancesFromRssiForEachBeacon()
                .doOnNext{ skald.debug("packetQueue - Beacons with Distance: $it") }
                .applyDistanceFilterToEachBeacon()
                .doOnNext{ skald.debug("packetQueue - Beacons with Distance after distance filter: $it") }
                .applyHumanWalkFiltering()
                .doOnNext{ skald.debug("packetQueue - Beacons with Distance after Walk filtering: $it") }
                .sortBeaconsByDistance()
                .doOnNext{ skald.debug("packetQueue - Beacons with Distance sorted by distance: $it") }
                .share()

        positioningHandle = sharedPacketQueue
                .doOnNext{ skald.debug("positioningHandle - attempting to process packets: $it") }
                .filter { it.isNotEmpty() }
                .doOnNext{ skald.debug("positioningHandle - not empty packets list: $it") }
                .calculateRawPositionFromNearbyBeacons()
                .doOnNext{ skald.debug("positioningHandle - calculated local position: $it") }
                .applyJumpRejection()
                .doOnNext{ skald.debug("positioningHandle - calculated local position after jump rejection: $it") }
                .applySmoothingFilter()
                .doOnNext{ skald.debug("positioningHandle - calculated local position after smoothing filter: $it") }
                .retryWhen {
                    it.flatMap { cause ->
                        skald.error("positioningHandle - Error occurred: $cause")
                        Flowable.just(true)
                    }
                }
                .subscribe({
                    mOnPositionUpdateListener?.onPositionUpdate(it)
                }, {
                    logger.error(it.message ?: "Unknown message")
                })

        outsideHandlingHandle = sharedPacketQueue
                .doOnNext{ skald.debug("outsideHandlingHandle - attempting to process packets: $it") }
                .checkIfPositionIsInsideOfLocation()
                .doOnNext{ skald.debug("outsideHandlingHandle - got location state: $it") }
                .checkLocationStateChange()
                .doOnNext{ skald.debug("outsideHandlingHandle - got location state change: $it") }
                .subscribe({
                    if (it == LocationStateChange.EXITED) {
                        mOnPositionUpdateListener?.onPositionOutsideLocation()
                    }
                }, { /* do nothing on error here*/ })
    }

    /**
     * This is probably the most important method here. Use it to pass scanned location data.
     * You can do the scanning using our BeaconManager class.
     */
    override fun onScannedLocationPackets(estimoteLocationPacket: EstimoteLocation) {
        skald.debug("onScannedLocationPackets called; estimoteLocationPacket: $estimoteLocationPacket")
        packetsQueue.onNext(estimoteLocationPacket.toScannedBeacon())
    }

    /**
     * In order to get estimated position from manager, you need to add listener to it.
     * Use this method to do this. Each consequent invocation of this function will override previous listener.
     */
    override fun setOnPositionUpdateListener(onPositionUpdateListener: OnPositionUpdateListener) {
        mOnPositionUpdateListener = onPositionUpdateListener
    }

    /**
     * Stops positioning algorithms. Make sure to use this method when you are not using positioning.
     */
    override fun stopPositioning() {
        mAnalyticsManager.generateEvent(mLocation.identifier, AnalyticsEventType.STOP_POSITIONING)
        motionDetectorDataSubscriber.unsubscribe()
        positioningHandle?.dispose()
        outsideHandlingHandle?.dispose()
        stepMotionDetector.stop()
    }

    private fun EstimoteLocation.toScannedBeacon() = ScannedBeacon(deviceId, rssi, channel, System.currentTimeMillis())

    private fun Beacon.toScannedBeacon() = ScannedBeacon(macAddress.address, rssi, timestamp = System.currentTimeMillis())

    private fun Observable<ScannedBeacon>.filterOutBeaconsFromOutsideOfLocation(): Observable<ScannedBeacon> =
            filter { scannedBeacon -> mLocation.beacons.any { it.beacon.mac.equals(scannedBeacon.id, ignoreCase = true) }}

    private fun Observable<ScannedBeacon>.keepBeaconPacketsBufferedByConfiguredTime(): Observable<List<ScannedBeacon>> =
            rollingBuffer(configFactory.getConfig().getBeaconSustainTimeSpan(), configFactory.getConfig().getBeaconSustainTimeUnit(), { hashSetOf<ScannedBeacon>() })
                    .buffer(mPositionUpdateIntervalMillis, TimeUnit.MILLISECONDS)
                    .map { it.flatMap { it.toList() }.distinctBy { it.id } }
                    .doOnNext { logger.debug("Attempting to calculate position based on beacons: ${it.map { it.id }}") }

    private fun Flowable<List<ScannedBeacon>>.mapToBeaconsWithRssiInLocation(): Flowable<List<BeaconWithRssi>> =
            map { mScannedBeaconToLocationBeaconMapper.mapScannedBeaconsToLocationBeacons(it, mLocation.beacons) }

    private fun Flowable<List<BeaconWithRssi>>.calculateDistancesFromRssiForEachBeacon(): Flowable<List<BeaconWithDistance>> =
            map { it.map { mRssiToDistanceCalculator.calculateDistance(it) } }

    private fun Flowable<List<BeaconWithDistance>>.applyHumanWalkFiltering(): Flowable<List<BeaconWithDistance>> =
            map { mHumanWalkStatefulDistanceFilter.filter(it) }

    private fun Flowable<List<BeaconWithDistance>>.applyDistanceFilterToEachBeacon(): Flowable<List<BeaconWithDistance>> =
            map { mBeaconDistanceFilter.filter(it) }

    private fun Flowable<List<BeaconWithDistance>>.calculateRawPositionFromNearbyBeacons(): Flowable<LocationPosition> =
            map { mPositionCalculator.calculatePosition(it) }

    private fun Flowable<LocationPosition>.applyJumpRejection(): Flowable<LocationPosition> =
            map { mJumpRejectionFilter.filter(it) }

    private fun Flowable<LocationPosition>.applySmoothingFilter(): Flowable<LocationPosition> =
            map { mPositionSmoothingFilter.filter(it) }

    private fun Flowable<List<BeaconWithDistance>>.sortBeaconsByDistance(): Flowable<List<BeaconWithDistance>> =
            map { it.sortedBy { it.distance } }

    private fun Flowable<List<BeaconWithDistance>>.checkIfPositionIsInsideOfLocation(): Flowable<LocationState> =
            buffer(3)
                    .map { it.flatMap { it.toList() } }
                    .map { if (it.isEmpty()) LocationState.OUTSIDE else LocationState.INSIDE }

    private fun Flowable<LocationState>.checkLocationStateChange(): Flowable<LocationStateChange> =
            distinctUntilChanged()
                    .map { if (it == LocationState.INSIDE) LocationStateChange.ENTERED else LocationStateChange.EXITED }

}

