package com.estimote.indoorsdk_module.algorithm.voronoi

import com.estimote.indoorsdk_module.algorithm.geometry.Line
import com.estimote.indoorsdk_module.algorithm.geometry.PlanarGeometry
import com.estimote.indoorsdk_module.algorithm.geometry.Point2D
import com.estimote.indoorsdk_module.cloud.Location
import com.estimote.indoorsdk_module.cloud.LocationPosition
import com.estimote.indoorsdk_module.cloud.LocationWall


/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class LocationPointTransformer(private val planarGeometry: PlanarGeometry) {

    private data class LineGeneralForm(val A: Double, val B: Double, val C: Double)

    fun findPointOnLocationWallClosestToPoint(locaton: Location, locationPosition: LocationPosition): LocationPosition {
        if (locaton.walls.isEmpty()) throw IllegalArgumentException("Unable to find closest point on location that doesn't have walls.")
        val closestPoint = locaton.walls
                .map { findRelationOfPointToLine(locationPosition.toPoint(), it.toLine()) }
                .minBy { it.first }!!.second
        return LocationPosition(closestPoint.x, closestPoint.y)
    }


    private fun findRelationOfPointToLine(point: Point2D, line: Line): Pair<Double, Point2D> {
        val lineGeneralForm = line.toGeneralForm()

        //  dist(A x + B y + C = 0, {x_0, y_0} ) = |A x_0 + B y_0 + C| / sqrt(A^2 + B^2)
        //  See: http://en.wikipedia.org/wiki/Distance_from_a_point_to_a_line
        val a2b2 = Math.pow(lineGeneralForm.A, 2.0) + Math.pow(lineGeneralForm.B, 2.0)
        val distanceNumerator = Math.abs(lineGeneralForm.A * point.x + lineGeneralForm.B * point.y + lineGeneralForm.C)
        val distance = distanceNumerator / Math.sqrt(a2b2)

        //  Closest point (x,y) to the line A x + B y + C = 0
        //  x = ( B ( B x_0 - A y_0 ) - A C ) / (A^2 + B^2)
        //  y = ( A ( -B x_0 + A y_0 ) - B C ) / (A^2 + B^2)
        //  See: http://en.wikipedia.org/wiki/Distance_from_a_point_to_a_line
        val xNumerator = lineGeneralForm.B * (lineGeneralForm.B * point.x - lineGeneralForm.A * point.y) - lineGeneralForm.A * lineGeneralForm.C
        val x = xNumerator / a2b2
        val yNumerator = lineGeneralForm.A * (-lineGeneralForm.B * point.x + lineGeneralForm.A * point.y) - lineGeneralForm.B * lineGeneralForm.C
        val y = yNumerator / a2b2

        // If closest point to line is not on the segment, we return distance to its one of its ends.
        // It's enough to check if it fits to wall's bounding box.
        if ((x < Math.min(line.first.x, line.second.x))
                || (y < Math.min(line.first.y, line.second.y))
                || (x > Math.max(line.first.x, line.second.x))
                || (y > Math.max(line.first.y, line.second.y))) {
            val distanceToLinePoint1 = planarGeometry.distanceBetweenPoints(point, line.first)
            val distanceToLinePoint2 = planarGeometry.distanceBetweenPoints(point, line.second)

            if (distanceToLinePoint1 < distanceToLinePoint2) {
                return (distanceToLinePoint1 to Point2D(line.first.x, line.first.y))
            } else {
                return (distanceToLinePoint2 to Point2D(line.second.x, line.second.y))
            }
        } else return (distance to Point2D(x, y))
    }

    private fun Line.toGeneralForm(): LineGeneralForm {
        if (this.first.x == this.second.x) {
            return LineGeneralForm(1.0, 0.0, -this.first.x)
        } else if (this.first.y == this.second.y) {
            return LineGeneralForm(0.0, 1.0, -this.first.y)
        } else {
            val m = (this.second.y - this.first.y) / (this.second.x - this.first.x)
            return LineGeneralForm(m, -1.0, -m * this.first.x + this.first.y)
        }
    }

    private fun LocationWall.toLine(): Line {
        return Line(Point2D(this.x1, this.y1), Point2D(this.x2, this.y2))
    }

    private fun LocationPosition.toPoint(): Point2D {
        return Point2D(this.x, this.y)
    }


}