package com.estimote.indoorsdk_module.analytics

import android.content.ContentValues
import android.content.Context
import android.database.Cursor
import android.database.sqlite.SQLiteDatabase
import android.database.sqlite.SQLiteOpenHelper
import java.util.ArrayList

/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class SQLiteAnalyticsEventRepository(context: Context) : AnalyticsEventRepository {

    private val eventsDbHelper: EventsDbHelper
    private var maxStoredEvents = MAX_STORED_EVENTS

    init {
        eventsDbHelper = EventsDbHelper(context)
    }

    companion object {
        private val MAX_STORED_EVENTS = 1000
        private val DB_NAME = "ESTIMOTE_INDOOR_ANALYTICS.db"
        private val EVENT_TABLE_NAME = "events"
        private val COLUMN_LOCATION_ID = "location_id"
        private val COLUMN_EVENT_NAME = "event_name"
        private val COLUMN_TIMESTAMP = "timestamp"
    }

    override fun save(event: AnalyticsEventEntity) {
        val db = eventsDbHelper.writableDatabase
        if (size() + 1 > maxStoredEvents) {
            db.delete(EVENT_TABLE_NAME, COLUMN_TIMESTAMP + " in (" + selectTop(1, COLUMN_TIMESTAMP) + ")", null)
        }
        db.insert(EVENT_TABLE_NAME, null, toContentValues(event))
    }

    private fun toContentValues(event: AnalyticsEventEntity): ContentValues {
        val contentValues = ContentValues()
        contentValues.put(COLUMN_EVENT_NAME, event.name)
        contentValues.put(COLUMN_LOCATION_ID, event.locationId)
        contentValues.put(COLUMN_TIMESTAMP, event.timestamp)
        return contentValues
    }

    override fun findOldest(limit: Int): List<AnalyticsEventEntity> {
        val events = ArrayList<AnalyticsEventEntity>()
        val db = eventsDbHelper.readableDatabase
        val cursor = db.rawQuery(selectTop(limit, "*"), null)
        while (cursor.moveToNext()) {
            events.add(createEventFrom(cursor))
        }
        cursor.close()
        return events
    }

    private fun selectTop(limit: Int, columns: String): String {
        return "select $columns from $EVENT_TABLE_NAME order by $COLUMN_TIMESTAMP asc limit $limit"
    }

    override fun deleteAllBefore(timestamp: Long) {
        val db = eventsDbHelper.writableDatabase
        db.delete(EVENT_TABLE_NAME, COLUMN_TIMESTAMP + " <= ?", arrayOf(timestamp.toString()))
    }

    private fun createEventFrom(cursor: Cursor): AnalyticsEventEntity {
        val timestamp = cursor.getLong(cursor.getColumnIndex(COLUMN_TIMESTAMP))
        val name = cursor.getString(cursor.getColumnIndex(COLUMN_EVENT_NAME))
        val locationId = cursor.getString(cursor.getColumnIndex(COLUMN_LOCATION_ID))
        return AnalyticsEventEntity(name, locationId, timestamp)
    }

    override fun size(): Int {
        val cursor = eventsDbHelper.readableDatabase.rawQuery("select count(*) from " + EVENT_TABLE_NAME, null)
        cursor.moveToNext()
        val count = cursor.getInt(0)
        cursor.close()
        return count
    }

    override fun setMaxStoredEvents(maxStoredEvents: Int) {
        this.maxStoredEvents = maxStoredEvents
    }

    private inner class EventsDbHelper(context: Context) : SQLiteOpenHelper(context, DB_NAME, null, 1) {

        override fun onCreate(db: SQLiteDatabase) {
            db.execSQL(
                    "CREATE TABLE " + EVENT_TABLE_NAME + " " +
                            "(" + COLUMN_EVENT_NAME + " text, " +
                            COLUMN_LOCATION_ID + " integer, " +
                            COLUMN_TIMESTAMP + " integer)"
            )
        }

        override fun onUpgrade(db: SQLiteDatabase, oldVersion: Int, newVersion: Int) {
            db.execSQL("DROP TABLE IF EXISTS events")
            onCreate(db)
        }
    }

}
