package com.estimote.indoorsdk_module.algorithm.position

import com.estimote.indoorsdk_module.algorithm.geometry.PlanarGeometry
import com.estimote.indoorsdk_module.algorithm.geometry.Point2D
import com.estimote.indoorsdk_module.cloud.LocationPosition
import com.estimote.indoorsdk_module.common.config.ConfigFactory


/**
 * Smooths new position according to previous state and parameters.
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class ExponentialPositionFilter(private val configFactory: ConfigFactory,
                                         private val planarGeometry: PlanarGeometry) : PositionFilter {

    private val START_POSITION = LocationPosition(-1.0, -1.0, 361.0)
    private var previousReportedPosition = START_POSITION
    private var previousSmoothedPosition = START_POSITION

    override fun filter(position: LocationPosition): LocationPosition {
        if (previousReportedPosition == START_POSITION) previousReportedPosition = position
        if (previousSmoothedPosition == START_POSITION) previousSmoothedPosition = position
        val newSmoothedPosition = smoothPosition(previousSmoothedPosition, position)
        return calculatePositionAccordingToDistance(newSmoothedPosition)
    }

    private fun calculatePositionAccordingToDistance(newSmoothedPosition: LocationPosition) : LocationPosition {
        val distanceChangeThreshold = configFactory.getConfig().getDistanceChangeThreshold()
        if (planarGeometry.distanceBetweenPoints(previousReportedPosition.toPoint(), newSmoothedPosition.toPoint()) > distanceChangeThreshold) {
            previousSmoothedPosition = newSmoothedPosition
            previousReportedPosition = previousSmoothedPosition
            return previousReportedPosition
        } else {
            previousSmoothedPosition = newSmoothedPosition
            previousReportedPosition = LocationPosition(previousReportedPosition.x, previousReportedPosition.y, newSmoothedPosition.orientation)
            return previousReportedPosition
        }
    }

    private fun smoothPosition(previousSmoothedPosition: LocationPosition, position: LocationPosition) : LocationPosition {
        val smoothingConstant = configFactory.getConfig().getExponentialSmoothingConstant()
        val smoothedX = smoothingConstant * previousSmoothedPosition.x + (1.0 - smoothingConstant) * position.x
        val smoothedY = smoothingConstant * previousSmoothedPosition.y + (1.0 - smoothingConstant) * position.y
        return LocationPosition(smoothedX, smoothedY, position.orientation)
    }

    private fun LocationPosition.toPoint(): Point2D {
        return Point2D(this.x, this.y)
    }
}