package com.estimote.indoorsdk_module.algorithm.distance

import com.estimote.indoorsdk_module.algorithm.model.BeaconWithDistance
import com.estimote.indoorsdk_module.common.config.Config
import com.estimote.indoorsdk_module.common.config.ConfigFactory

/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class HumanWalkStatefulDistanceCalculator(private val configFactory: ConfigFactory) : StatefulDistanceCalculator {

    override fun calculateDistance(previousDistance: BeaconWithDistance?, newDistance: BeaconWithDistance): BeaconWithDistance {
        val config = configFactory.getConfig()
        return if (previousDistance == null) {
            newDistance
        } else {
            calculateNewDistanceWithHumanWalkCorrection(newDistance, previousDistance, config)
        }
    }

    private fun calculateNewDistanceWithHumanWalkCorrection(newData: BeaconWithDistance, previousData: BeaconWithDistance, config: Config): BeaconWithDistance {
        val distanceDifference = newData.distance - previousData.distance
        val timeDifference = newData.timestamp - previousData.timestamp
        return if (Math.abs(distanceDifference) < config.getDistanceJumpEnabledThreshold()) {
            BeaconWithDistance(newData.beacon, newData.distance + getHumanWalkDistanceCorrection(distanceDifference, timeDifference, config), newData.timestamp)
        } else {
            BeaconWithDistance(newData.beacon, newData.distance, newData.timestamp)
        }
    }

    private fun getHumanWalkDistanceCorrection(distanceDifference: Double, timeDifference: Long, config: Config): Double {
        return if (timeDifference > 0) {
            val smoothedDistance = distanceDifference * config.getExponentialSmoothingConstant()
            val maxDifference = config.getHumanVelocityMaxMultiplier() * timeDifference * config.getNormalHumanVelocity()
            Math.max(-maxDifference, Math.min(maxDifference, smoothedDistance))
        } else {
            distanceDifference
        }
    }

}