package com.estimote.indoorsdk_module.algorithm.voronoi;

/*
 * Copyright (c) 2005, 2007 by L. Paul Chew.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

import android.support.annotation.NonNull;
/**
 * Points in Euclidean space, implemented as double[].
 * <p/>
 * Includes simple geometric operations.
 * Uses matrices; a matrix is represented as an array of Pnts.
 * Uses simplices; a simplex is represented as an array of Pnts.
 *
 * @author Paul Chew
 */
class Point implements Comparable<Point> {

  private final double[] coords;

  public Point(double... coords) {
    this.coords = coords.clone();
  }

  @Override
  public boolean equals(Object other) {
    if (!(other instanceof Point)) return false;
    Point p = (Point) other;
    if (this.coords.length != p.coords.length) return false;
    for (int i = 0; i < this.coords.length; i++)
      if (this.coords[i] != p.coords[i]) return false;
    return true;
  }

  @Override
  public int hashCode() {
    int hash = 0;
    for (double c : this.coords) {
      long bits = Double.doubleToLongBits(c);
      hash = (31 * hash) ^ (int) (bits ^ (bits >> 32));
    }
    return hash;
  }

  @Override
  public String toString() {
    return "(" + x() + ", " + y() + ")";
  }

  /* Pnts as vectors */

  /**
   * @return the specified coordinate of this Point
   * @throws ArrayIndexOutOfBoundsException for bad coordinate
   */
  public double coord(int i) {
    return this.coords[i];
  }

  public double x() {
    return coords[0];
  }

  public double y() {
    return coords[1];
  }

  /**
   * @return this Point's dimension.
   */
  public int dimension() {
    return coords.length;
  }

  /**
   * Check that dimensions match.
   *
   * @param p the Point to check (against this Point)
   * @return the dimension of the Pnts
   * @throws IllegalArgumentException if dimension fail to match
   */
  public int dimCheck(Point p) {
    int len = this.coords.length;
    if (len != p.coords.length)
      throw new IllegalArgumentException("Dimension mismatch");
    return len;
  }

  /**
   * Create a new Point by adding additional coords to this Point.
   *
   * @param coords the new coords (added on the right end)
   * @return a new Point with the additional coords
   */
  public Point extend(double... coords) {
    double[] result = new double[this.coords.length + coords.length];
    System.arraycopy(this.coords, 0, result, 0, this.coords.length);
    System.arraycopy(coords, 0, result, this.coords.length, coords.length);
    return new Point(result);
  }

  /**
   * Dot product.
   *
   * @param p the other Point
   * @return dot product of this Point and p
   */
  public double dot(Point p) {
    int len = dimCheck(p);
    double sum = 0;
    for (int i = 0; i < len; i++)
      sum += this.coords[i] * p.coords[i];
    return sum;
  }

  /**
   * Magnitude (as a vector).
   *
   * @return the Euclidean length of this vector
   */
  public double magnitude() {
    return Math.sqrt(this.dot(this));
  }

  /**
   * Subtract.
   *
   * @param p the other Point
   * @return a new Point = this - p
   */
  public Point subtract(Point p) {
    int len = dimCheck(p);
    double[] coords = new double[len];
    for (int i = 0; i < len; i++)
      coords[i] = this.coords[i] - p.coords[i];
    return new Point(coords);
  }

  /**
   * Add.
   *
   * @param p the other Point
   * @return a new Point = this + p
   */
  public Point add(Point p) {
    int len = dimCheck(p);
    double[] coords = new double[len];
    for (int i = 0; i < len; i++)
      coords[i] = this.coords[i] + p.coords[i];
    return new Point(coords);
  }

  /**
   * Angle (in radians) between two Pnts (treated as vectors).
   *
   * @param p the other Point
   * @return the angle (in radians) between the two Pnts
   */
  public double angle(Point p) {
    return Math.acos(this.dot(p) / (this.magnitude() * p.magnitude()));
  }

  /**
   * Perpendicular bisector of two Pnts.
   * Works in any dimension.  The coefficients are returned as a Point of one
   * higher dimension (e.g., (A,B,C,D) for an equation of the form
   * Ax + By + Cz + D = 0).
   *
   * @param point the other point
   * @return the coefficients of the perpendicular bisector
   */
  public Point bisector(Point point) {
    dimCheck(point);
    Point diff = this.subtract(point);
    Point sum = this.add(point);
    double dot = diff.dot(sum);
    return diff.extend(-dot / 2);
  }

    /* Pnts as matrices */

  /**
   * Compute the determinant of a matrix (array of Pnts).
   * This is not an efficient implementation, but should be adequate
   * for low dimension.
   *
   * @param matrix the matrix as an array of Pnts
   * @return the determinnant of the input matrix
   * @throws IllegalArgumentException if dimensions are wrong
   */
  public static double determinant(Point[] matrix) {
    if (matrix.length != matrix[0].dimension())
      throw new IllegalArgumentException("Matrix is not square");
    boolean[] columns = new boolean[matrix.length];
    for (int i = 0; i < matrix.length; i++) columns[i] = true;
    try {
      return determinant(matrix, 0, columns);
    } catch (ArrayIndexOutOfBoundsException e) {
      throw new IllegalArgumentException("Matrix is wrong shape");
    }
  }

  /**
   * Compute the determinant of a submatrix specified by starting row
   * and by "active" columns.
   *
   * @param matrix  the matrix as an array of Pnts
   * @param row     the starting row
   * @param columns a boolean array indicating the "active" columns
   * @return the determinant of the specified submatrix
   * @throws ArrayIndexOutOfBoundsException if dimensions are wrong
   */
  private static double determinant(Point[] matrix, int row, boolean[] columns) {
    if (row == matrix.length) return 1;
    double sum = 0;
    int sign = 1;
    for (int col = 0; col < columns.length; col++) {
      if (!columns[col]) continue;
      columns[col] = false;
      sum += sign * matrix[row].coords[col] *
          determinant(matrix, row + 1, columns);
      columns[col] = true;
      sign = -sign;
    }
    return sum;
  }

  /**
   * Compute generalized cross-product of the rows of a matrix.
   * The result is a Point perpendicular (as a vector) to each row of
   * the matrix.  This is not an efficient implementation, but should
   * be adequate for low dimension.
   *
   * @param matrix the matrix of Pnts (one less row than the Point dimension)
   * @return a Point perpendicular to each row Point
   * @throws IllegalArgumentException if matrix is wrong shape
   */
  public static Point cross(Point[] matrix) {
    int len = matrix.length + 1;
    if (len != matrix[0].dimension())
      throw new IllegalArgumentException("Dimension mismatch");
    boolean[] columns = new boolean[len];
    for (int i = 0; i < len; i++) columns[i] = true;
    double[] result = new double[len];
    int sign = 1;
    try {
      for (int i = 0; i < len; i++) {
        columns[i] = false;
        result[i] = sign * determinant(matrix, 0, columns);
        columns[i] = true;
        sign = -sign;
      }
    } catch (ArrayIndexOutOfBoundsException e) {
      throw new IllegalArgumentException("Matrix is wrong shape");
    }
    return new Point(result);
  }

    /* Pnts as simplices */

  /**
   * Determine the signed content (i.e., area or volume, etc.) of a simplex.
   *
   * @param simplex the simplex (as an array of Pnts)
   * @return the signed content of the simplex
   */
  public static double content(Point[] simplex) {
    Point[] matrix = new Point[simplex.length];
    for (int i = 0; i < matrix.length; i++)
      matrix[i] = simplex[i].extend(1);
    int fact = 1;
    for (int i = 1; i < matrix.length; i++) fact = fact * i;
    return determinant(matrix) / fact;
  }

  /**
   * Relation between this Point and a simplex (represented as an array of
   * Pnts). Result is an array of signs, one for each vertex of the simplex,
   * indicating the relation between the vertex, the vertex's opposite facet,
   * and this Point.
   * <p/>
   * <pre>
   *   -1 means Point is on same side of facet
   *    0 means Point is on the facet
   *   +1 means Point is on opposite side of facet
   * </pre>
   *
   * @param simplex an array of Pnts representing a simplex
   * @return an array of signs showing relation between this Point and simplex
   */
  public int[] relation(Point[] simplex) {
        /* In 2D, we compute the cross of this matrix:
         *    1   1   1   1
         *    p0  a0  b0  c0
         *    p1  a1  b1  c1
         * where (a, b, c) is the simplex and p is this Point. The result is a
         * vector in which the first coordinate is the signed area (all signed
         * areas are off by the same constant factor) of the simplex and the
         * remaining coords are the *negated* signed areas for the
         * simplices in which p is substituted for each of the vertices.
         * Analogous results occur in higher dimensions.
         */
    int dim = simplex.length - 1;
    if (this.dimension() != dim)
      throw new IllegalArgumentException("Dimension mismatch");

        /* Create and load the matrix */
    Point[] matrix = new Point[dim + 1];
        /* First row */
    double[] coords = new double[dim + 2];
    for (int j = 0; j < coords.length; j++) coords[j] = 1;
    matrix[0] = new Point(coords);
        /* Other rows */
    for (int i = 0; i < dim; i++) {
      coords[0] = this.coords[i];
      for (int j = 0; j < simplex.length; j++)
        coords[j + 1] = simplex[j].coords[i];
      matrix[i + 1] = new Point(coords);
    }

        /* Compute and analyze the vector of areas/volumes/contents */
    Point vector = cross(matrix);
    double content = vector.coords[0];
    int[] result = new int[dim + 1];
    for (int i = 0; i < result.length; i++) {
      double value = vector.coords[i + 1];
      if (Math.abs(value) <= 1.0e-6 * Math.abs(content)) result[i] = 0;
      else if (value < 0) result[i] = -1;
      else result[i] = 1;
    }
    if (content < 0) {
      for (int i = 0; i < result.length; i++)
        result[i] = -result[i];
    }
    if (content == 0) {
      for (int i = 0; i < result.length; i++)
        result[i] = Math.abs(result[i]);
    }
    return result;
  }

  /**
   * Test if this Point is outside of simplex.
   *
   * @param simplex the simplex (an array of Pnts)
   * @return simplex Point that "witnesses" outsideness (or null if not outside)
   */
  public Point isOutside(Point[] simplex) {
    int[] result = this.relation(simplex);
    for (int i = 0; i < result.length; i++) {
      if (result[i] > 0) return simplex[i];
    }
    return null;
  }

  /**
   * Test if this Point is on a simplex.
   *
   * @param simplex the simplex (an array of Pnts)
   * @return the simplex Point that "witnesses" on-ness (or null if not on)
   */
  public Point isOn(Point[] simplex) {
    int[] result = this.relation(simplex);
    Point witness = null;
    for (int i = 0; i < result.length; i++) {
      if (result[i] == 0) witness = simplex[i];
      else if (result[i] > 0) return null;
    }
    return witness;
  }

  /**
   * Test if this Point is inside a simplex.
   *
   * @param simplex the simplex (an arary of Pnts)
   * @return true iff this Point is inside simplex.
   */
  public boolean isInside(Point[] simplex) {
    int[] result = this.relation(simplex);
    for (int r : result) if (r >= 0) return false;
    return true;
  }

  /**
   * Test relation between this Point and circumcircle of a simplex.
   *
   * @param simplex the simplex (as an array of Pnts)
   * @return -1, 0, or +1 for inside, on, or outside of circumcircle
   */
  public int vsCircumcircle(Point[] simplex) {
    Point[] matrix = new Point[simplex.length + 1];
    for (int i = 0; i < simplex.length; i++)
      matrix[i] = simplex[i].extend(1, simplex[i].dot(simplex[i]));
    matrix[simplex.length] = this.extend(1, this.dot(this));
    double d = determinant(matrix);
    int result = (d < 0) ? -1 : ((d > 0) ? +1 : 0);
    if (content(simplex) < 0) result = -result;
    return result;
  }

  /**
   * Circumcenter of a simplex.
   *
   * @param simplex the simplex (as an array of Pnts)
   * @return the circumcenter (a Point) of simplex
   */
  public static Point circumcenter(Point[] simplex) {
    int dim = simplex[0].dimension();
    if (simplex.length - 1 != dim)
      throw new IllegalArgumentException("Dimension mismatch");
    Point[] matrix = new Point[dim];
    for (int i = 0; i < dim; i++)
      matrix[i] = simplex[i].bisector(simplex[i + 1]);
    Point hCenter = cross(matrix);      // Center in homogeneous coords
    double last = hCenter.coords[dim];
    double[] result = new double[dim];
    for (int i = 0; i < dim; i++) result[i] = hCenter.coords[i] / last;
    return new Point(result);
  }

  @Override
  public int compareTo(@NonNull Point o) {
    if (this.x() == o.x()) {
      return (int) (this.y() - o.y());
    } else {
      return (int)(this.x() - o.x());
    }
  }
}