package com.estimote.indoorsdk_module.algorithm

import com.estimote.indoorsdk_module.algorithm.distance.RssiToDistanceCalculator
import com.estimote.indoorsdk_module.algorithm.filter.DistanceFilter
import com.estimote.indoorsdk_module.algorithm.filter.HumanWalkStatefulDistanceFilter
import com.estimote.indoorsdk_module.algorithm.filter.ScannedBeaconToLocationBeaconMapper
import com.estimote.indoorsdk_module.algorithm.model.BeaconWithDistance
import com.estimote.indoorsdk_module.algorithm.model.BeaconWithRssi
import com.estimote.indoorsdk_module.algorithm.model.ScannedBeacon
import com.estimote.indoorsdk_module.algorithm.position.ExponentialPositionFilter
import com.estimote.indoorsdk_module.algorithm.position.JumpRejectionFilter
import com.estimote.indoorsdk_module.algorithm.position.PositionCalculator
import com.estimote.indoorsdk_module.algorithm.sensor.StepMotionDetectorSubscriber
import com.estimote.indoorsdk_module.analytics.AnalyticsEventType
import com.estimote.indoorsdk_module.analytics.AnalyticsManager
import com.estimote.indoorsdk_module.cloud.Location
import com.estimote.indoorsdk_module.cloud.LocationPosition
import com.estimote.indoorsdk_module.common.state.LocationState
import com.estimote.indoorsdk_module.common.state.LocationStateChange
import com.estimote.indoorsdk_module.common.threading.SchedulerProvider
import com.estimote.indoorsdk_module.logging.IndoorLogger
import com.estimote.internal_plugins_api.scanning.Beacon
import com.estimote.internal_plugins_api.scanning.EstimoteLocation
import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import io.reactivex.Observable
import io.reactivex.subjects.PublishSubject
import java.util.concurrent.TimeUnit
import javax.inject.Inject


/**
 * This is the manager which is responsible of all the positioning algorithms.
 * You need to provide it with scanned beacon data using method onScannedPackets
 *
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class EstimoteIndoorLocationManager @Inject constructor(private val mLocation: Location,
                                                                 private val mPositionUpdateIntervalMillis: Long,
                                                                 private val motionDetectorDataSubscriber: StepMotionDetectorSubscriber,
                                                                 private val mScannedBeaconToLocationBeaconMapper: ScannedBeaconToLocationBeaconMapper,
                                                                 private val mRssiToDistanceCalculator: RssiToDistanceCalculator,
                                                                 private val mHumanWalkStatefulDistanceFilter: HumanWalkStatefulDistanceFilter,
                                                                 private val mBeaconDistanceFilter: DistanceFilter,
                                                                 private val mPositionCalculator: PositionCalculator,
                                                                 private val mJumpRejectionFilter: JumpRejectionFilter,
                                                                 private val mPositionSmoothingFilter: ExponentialPositionFilter,
                                                                 private val mAnalyticsManager: AnalyticsManager,
                                                                 private val mSchedulers: SchedulerProvider,
                                                                 private val logger: IndoorLogger) : IndoorLocationManager {

    private var mOnPositionUpdateListener: OnPositionUpdateListener? = null
    private var packetsQueue = PublishSubject.create<ScannedBeacon>()

    init {
        // scan data handling
        val sharedPacketQueue = packetsQueue
                .bufferSingleScansIntoBatches()
                .toFlowable(BackpressureStrategy.LATEST)
                .observeOn(mSchedulers.computation())
                .findBeaconsFromLocation()
                .calculateDistancesFromRssiForEachBeacon()
                .applyDistanceFilterToEachBeacon()
                .applyHumanWalkFiltering()
                .sortBeaconsByDistance()
                .share()

        // location position handling
        sharedPacketQueue
                .filter { it.isNotEmpty() }
                .calculateRawPositionFromNearbyBeacons()
                .applyJumpRejection()
                .applySmoothingFilter()
                .subscribe({
                    mOnPositionUpdateListener?.onPositionUpdate(it)
                }, { logger.error(it.message ?: "Unknown message") })

        // outside location handling
        sharedPacketQueue
                .checkIfPositionIsInsideOfLocation()
                .checkLocationStateChange()
                .subscribe({
                    if (it == LocationStateChange.EXITED) {
                        mOnPositionUpdateListener?.onPositionOutsideLocation()
                    }
                }, { /* do nothing on error here*/ })
    }

    /**
     * Notifies Manager to start it's positioning algorithms. Make sure to call this method before starting scanning.
     */
    override fun startPositioning() {
        mAnalyticsManager.generateEvent(mLocation.identifier, AnalyticsEventType.START_POSITIONING)
        motionDetectorDataSubscriber.subscribe()
    }

    /**
     * This is probably the most important method here. Use it to pass scanned location data.
     * You can do the scanning using our BeaconManager class.
     */
    override fun onScannedLocationPackets(estimoteLocationPacket: EstimoteLocation) {
        packetsQueue.onNext(estimoteLocationPacket.toScannedBeacon())
    }

    /**
     * This is probably the most important method here. Use it to pass scanned beacon data.
     * You can do the scanning using our BeaconManager class.
     */
    override fun onScannedBeaconPackets(beaconPacket: Beacon) {
        packetsQueue.onNext(beaconPacket.toScannedBeacon())
    }

    /**
     * In order to get estimated position from manager, you need to add listener to it.
     * Use this method to do this. Each consequent invocation of this function will override previous listener.
     */
    override fun setOnPositionUpdateListener(onPositionUpdateListener: OnPositionUpdateListener) {
        mOnPositionUpdateListener = onPositionUpdateListener
    }

    /**
     * Stops positioning algorithms. Make sure to use this method when you are not using positioning.
     */
    override fun stopPositioning() {
        mAnalyticsManager.generateEvent(mLocation.identifier, AnalyticsEventType.STOP_POSITIONING)
        motionDetectorDataSubscriber.unsubscribe()
    }

    private fun EstimoteLocation.toScannedBeacon() = ScannedBeacon(deviceId, rssi, channel, System.currentTimeMillis())

    private fun Beacon.toScannedBeacon() = ScannedBeacon(macAddress.address, rssi, timestamp = System.currentTimeMillis())

    private fun Flowable<List<ScannedBeacon>>.findBeaconsFromLocation(): Flowable<List<BeaconWithRssi>> =
            map { mScannedBeaconToLocationBeaconMapper.mapScannedBeaconsToLocationBeacons(it, mLocation.beacons) }

    private fun Flowable<List<BeaconWithRssi>>.calculateDistancesFromRssiForEachBeacon(): Flowable<List<BeaconWithDistance>> =
            map { it.map { mRssiToDistanceCalculator.calculateDistance(it) } }

    private fun Flowable<List<BeaconWithDistance>>.applyHumanWalkFiltering(): Flowable<List<BeaconWithDistance>> =
            map { mHumanWalkStatefulDistanceFilter.filter(it) }

    private fun Flowable<List<BeaconWithDistance>>.applyDistanceFilterToEachBeacon(): Flowable<List<BeaconWithDistance>> =
            map { mBeaconDistanceFilter.filter(it) }

    private fun Flowable<List<BeaconWithDistance>>.calculateRawPositionFromNearbyBeacons(): Flowable<LocationPosition> =
            map { mPositionCalculator.calculatePosition(it) }

    private fun Flowable<LocationPosition>.applyJumpRejection(): Flowable<LocationPosition> =
            map { mJumpRejectionFilter.filter(it) }

    private fun Flowable<LocationPosition>.applySmoothingFilter(): Flowable<LocationPosition> =
            map { mPositionSmoothingFilter.filter(it) }

    private fun Observable<ScannedBeacon>.bufferSingleScansIntoBatches(): Observable<List<ScannedBeacon>> =
            buffer(mPositionUpdateIntervalMillis, 1000, TimeUnit.MILLISECONDS, mSchedulers.computation(), { HashSet<ScannedBeacon>() })
                    .map { it.toList() }
                    .doOnNext { logger.debug("Buffered ${it.size} beacons") }

    private fun Flowable<List<BeaconWithDistance>>.sortBeaconsByDistance(): Flowable<List<BeaconWithDistance>> =
            map { it.sortedBy { it.distance } }


    private fun Flowable<List<BeaconWithDistance>>.checkIfPositionIsInsideOfLocation(): Flowable<LocationState> =
            map { if (it.isEmpty()) LocationState.OUTSIDE else LocationState.INSIDE }


    private fun Flowable<LocationState>.checkLocationStateChange(): Flowable<LocationStateChange> =
            distinctUntilChanged()
                    .map { if (it == LocationState.INSIDE) LocationStateChange.ENTERED else LocationStateChange.EXITED  }

}

