package com.estimote.indoorsdk_module.analytics

import com.estimote.indoorsdk_module.cloud.AnalyticsEvent
import com.estimote.indoorsdk_module.cloud.AnalyticsEventHeader
import com.estimote.indoorsdk_module.cloud.analytics.AnalyticsCloudManager
import com.estimote.indoorsdk_module.common.threading.SchedulerProvider
import com.estimote.indoorsdk_module.common.time.Timer
import com.estimote.indoorsdk_module.common.time.DateFormatter
import io.reactivex.processors.PublishProcessor

/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class IndoorAnalyticsManager(private val numberOfEventsToSendAtOnce: Int,
                                      private val generalInfo: GeneralInfo,
                                      private val eventRepository: AnalyticsEventRepository,
                                      private val analyticsCloudManager: AnalyticsCloudManager,
                                      private val dateFormatter: DateFormatter,
                                      private val timer: Timer,
                                      schedulers: SchedulerProvider) : AnalyticsManager {

    private val queueSubject = PublishProcessor.create<() -> Unit>()

    init {
        queueSubject.onBackpressureBuffer().subscribeOn(schedulers.io()).subscribe { it() }
    }

    override fun generateEvent(locationId: String, eventType: AnalyticsEventType) {
        queueSubject.onNext {
            eventRepository.save(AnalyticsEventEntity(eventType.text, locationId, timer.getEpochTimeMillis()))
            sendEventsIfLimitReached()
        }
    }

    private fun sendEventsIfLimitReached() {
        if (eventRepository.size() >= numberOfEventsToSendAtOnce) {
            takeChunkOfEvents()
                    .convertToCloudRequestObject()
                    .sendChunkAndRepeatUntilNoEventsLeft()
        }
    }

    private fun takeChunkOfEvents(): List<AnalyticsEventEntity> {
        return eventRepository.findOldest(numberOfEventsToSendAtOnce)
    }

    private fun List<AnalyticsEventEntity>.convertToCloudRequestObject(): Pair<AnalyticsEventHeader, Long> {
        val lastSentEventTimestamp = this.last().timestamp
        val events = this.map { it.toAnalyticsEvent() }
        return AnalyticsEventHeader(generalInfo.deviceModel, generalInfo.sdkVersion, generalInfo.platform, events) to lastSentEventTimestamp
    }

    private fun Pair<AnalyticsEventHeader, Long>.sendChunkAndRepeatUntilNoEventsLeft() {
        analyticsCloudManager.getPositioningParameters(this.first)
                .subscribe {
                    eventRepository.deleteAllBefore(this.second)
                    if (eventRepository.size() != 0) sendEventsIfLimitReached()
                }
    }

    private fun AnalyticsEventEntity.toAnalyticsEvent(): AnalyticsEvent {
        return AnalyticsEvent(
                this.name,
                generalInfo.appName,
                generalInfo.deviceId,
                generalInfo.deviceModel,
                generalInfo.sdkVersion,
                generalInfo.platform,
                generalInfo.locationCloudId,
                dateFormatter.millisToDate(this.timestamp))
    }

}