package com.estimote.indoorsdk_module.cloud

import android.support.annotation.Keep
import com.google.gson.annotations.SerializedName

/**
 * Data class for basic Location information.
 * All coordinates for walls, points and beacons are here.
 * You can manage your Location using Estimote Cloud account
 * See https://cloud.estimote.com/
 */
@Keep
data class Location(
        /** Name of the location */
        val name: String = "",
        /** Location unique identifier */
        val identifier: String = "",
        /** If this location is accessible publicly, or only by the given Estimote Cloud account */
        val public: Boolean = true,
        /** Name of location owner */
        val owner: String = "",
        /** Location global orientation. If not supplied, the default value is `0.0` */
        val orientation: Double = 0.0,
        /** List containing location wall objects (with coordinates and stuff) */
        val walls: List<LocationWall> = emptyList(),
        /** List containing location beacon objects (with coordinates and stuff) */
        val beacons: List<LocationBeacon> = emptyList(),
        /** List containing location linear objects - doors, windows, etc. */
        val linearObjects: List<LocationLinearObject> = emptyList()
)

/**
 * Data class for location wall model. Wall is defined by two points.
 */
@Keep
data class LocationWall(
        /** X coordinate of the first point */
        val x1: Double,
        /** Y coordinate of the first point */
        val y1: Double,
        /** X coordinate of the second point */
        val x2: Double,
        /** Y coordinate of the second point */
        val y2: Double,
        /** Wall orientation vector */
        val orientation: Double)

/**
 * Data class for location beacon model.
 * Contains beacon specific data and [LocationPosition] with coordinates.
 * @see [LocationBeaconData]
 * @see [LocationPosition]
 */
@Keep
data class LocationBeacon(
        /** Beacon specific data */
        val beacon: LocationBeaconData,
        /** Beacon coordinates */
        val position: LocationPosition)

/**
 * Data class for location beacon specific data.
 */
@Keep
data class LocationBeaconData(
        /** Beacon mac address or Estimote Identifier (New beacons) */
        val mac: String,
        /**
         * Beacon color defined by enum class.
         * @see LocationBeaconColor
         */
        val color: LocationBeaconColor)

/**
 * Location position data class. Contains x,y coordinates and orientation.
 */
@Keep
data class LocationPosition(
        /** X coordinate of the point */
        val x: Double = 0.0,
        /** Y coordinate of the second point */
        val y: Double = 0.0,
        /** Orientation means value from 360 degrees with 0.0 pointing toward device upper screen border. Clockwise.*/
        val orientation: Double = 0.0)

/**
 * Location linear object data - such as walls, windows, etc. Specified by [type].
 */
@Keep
data class LocationLinearObject(
        /** X coordinate of the first point */
        val x1: Double,
        /** Y coordinate of the first point */
        val y1: Double,
        /** X coordinate of the second point */
        val x2: Double,
        /** Y coordinate of the second point */
        val y2: Double,
        /** Orientation means value from 360 degrees with 0.0 pointing toward device upper screen border. Clockwise.*/
        val orientation: Double,
        /** Linear object type id */
        val type: Int)

internal data class PositioningParameters(val beaconParameters: BeaconParameters)

internal data class BeaconParameters(val measuredPower: Int)

internal data class AnalyticsEventHeader(
        @SerializedName("device_model") val deviceModel: String,
        @SerializedName("sdk_version") val sdkVersion: String,
        @SerializedName("platform") val platform: String,
        @SerializedName("events") val events: List<AnalyticsEvent>)

internal data class AnalyticsEvent(
        @SerializedName("event") val name: String,
        @SerializedName("app_name") val appName: String = "",
        @SerializedName("device_id") val deviceId: String = "",
        @SerializedName("device_model") val deviceModel: String = "",
        @SerializedName("sdk_version") val sdkVersion: String = "",
        @SerializedName("platform") val platform: String = "",
        @SerializedName("location_identifier") val locationId: String,
        @SerializedName("date") val timestamp: String)

/**
 * Location beacon color enum value.
 */
@Keep
enum class LocationBeaconColor(val text: String) {
    @SerializedName("unknown")
    UNKNOWN("Unknown"),
    @SerializedName("mint")
    MINT_COCKTAIL("Mint Cocktail"),
    @SerializedName("ice")
    ICY_MARSHMALLOW("Icy Marshmallow"),
    @SerializedName("blueberry")
    BLUEBERRY_PIE("Blueberry Pie"),
    @SerializedName("beetroot")
    SWEET_BEETROOT("Sweet Beetroot"),
    @SerializedName("candy")
    CANDY_FLOSS("Candy Floss"),
    @SerializedName("lemon")
    LEMON_TART("Lemon Tart"),
    @SerializedName("vanilla")
    VANILLA_JELLO("Vanilla Jello"),
    @SerializedName("liquorice")
    LIQUORICE_SWIRL("Liquorice Swirl"),
    @SerializedName("white")
    WHITE("White"),
    @SerializedName("black")
    BLACK("Black"),
    @SerializedName("grey")
    GREY("Grey"),
    @SerializedName("transparent")
    TRANSPARENT("Transparent");

    override fun toString(): String {
        return this.text
    }


}
