package com.estimote.indoorsdk_module.algorithm.voronoi

import com.estimote.indoorsdk_module.algorithm.geometry.PlanarGeometry
import com.estimote.indoorsdk_module.algorithm.geometry.Point2D
import com.estimote.indoorsdk_module.algorithm.geometry.Polygon
import com.estimote.indoorsdk_module.algorithm.geometry.Triangle2D
import com.estimote.indoorsdk_module.cloud.*
import com.estimote.indoorsdk_module.common.config.ConfigFactory
import com.estimote.indoorsdk_module.common.extensions.getMaxX
import com.estimote.indoorsdk_module.common.extensions.getMaxY
import com.estimote.indoorsdk_module.common.extensions.getMinX
import com.estimote.indoorsdk_module.common.extensions.getMinY
import java.util.*


/**
 * @author Pawel Dylag (pawel.dylag@estimote.com)
 */
internal class VoronoiSplitter(private val configFactory: ConfigFactory,
                               private val planarGeometry: PlanarGeometry,
                               private val locationPointTransformer: LocationPointTransformer) {

    private val TRIANGLE_AROUND_LOCATION_THRESHOLD = 10

    private lateinit var initialTriangleAroundLocation: Triangle

    fun getVoronoiCentersForBeaconsInLocation(location: Location): Map<LocationBeacon, LocationPosition> {
        if (location.beacons.isEmpty()) throw IllegalArgumentException("Unable to calculate Voronoi for location with no beacons.")
        if (location.beacons.size == 1) return centerOfLocationWithOneBeacon(location)
        return prepareTriangulationForLocation(location, TRIANGLE_AROUND_LOCATION_THRESHOLD)
                .putBeaconsAsTriangulationPoints(location.beacons)
                .calculateVoronoiCells()
                .clipVoronoiCellsToConvexHullOf(location)
                .calculateVoronoiCellCenters()
                .ifAnyPointIsOutsideTheLocationThenClipIt(location)
                .andMapThemToClosestBeaconIn(location)
    }

    private fun prepareTriangulationForLocation(location: Location, threshold: Int): Triangulation {
        val minPoint = Point2D(location.getMinX(), location.getMinY())
        val maxPoint = Point2D(location.getMaxX(), location.getMaxY())
        val triangle2D = planarGeometry.calculateTriangleContainingPoints(minPoint, maxPoint, threshold)
        initialTriangleAroundLocation = triangleFromTriangle2D(triangle2D)
        return Triangulation(initialTriangleAroundLocation)
    }

    private fun Triangulation.putBeaconsAsTriangulationPoints(locationBeacons: List<LocationBeacon>): Triangulation {
        for (beacon in locationBeacons) {
            delaunayPlace(beacon.toPoint().toDelaunayPoint())
        }
        return this
    }

    private fun Triangulation.calculateVoronoiCells(): List<Polygon> {
        val alreadyProcessedSites = HashSet<Point>()
        alreadyProcessedSites.addAll(initialTriangleAroundLocation)
        val mutableListOfCells = mutableListOf<List<Point>>()
        for (triangle in this) {
            for (site in triangle) {
                if (alreadyProcessedSites.contains(site)) {
                    continue
                }
                alreadyProcessedSites.add(site)
                val cell = surroundingTriangles(site, triangle).map { it.circumcenter }
                mutableListOfCells.add(cell)
            }
        }
        return mutableListOfCells.toPolygonList()
    }

    private fun List<Polygon>.clipVoronoiCellsToConvexHullOf(location: Location): List<Polygon> {
        val epsilon = configFactory.getConfig().getEpsilon()
        return map {
            val cellConvexHull = planarGeometry.calculateConvexHullOfPolygon(it)
            val cellClippedToLocation = planarGeometry.intersectionOfPolygons(cellConvexHull, location.getWallPoints(), epsilon)
            return@map cellClippedToLocation
        }
    }

    private fun List<Polygon>.calculateVoronoiCellCenters(): List<LocationPosition> {
        return map {
            val center = planarGeometry.calculateCentroidOfPolygon(it)
            return@map LocationPosition(center.x, center.y, 0.0)
        }
    }

    private fun List<LocationPosition>.andMapThemToClosestBeaconIn(location: Location): Map<LocationBeacon, LocationPosition> {
        return associate { location.findBeaconClosestToPoint(it) to it }
    }

    private fun centerOfLocationWithOneBeacon(location: Location): Map<LocationBeacon, LocationPosition> {
        val locationConvexHull = planarGeometry.calculateConvexHullOfPolygon(location.getWallPoints())
        val voronoiCenter = planarGeometry.calculateCentroidOfPolygon(locationConvexHull).toLocationPosition()
        return mapOf(location.beacons[0] to voronoiCenter)
    }


    private fun List<LocationPosition>.ifAnyPointIsOutsideTheLocationThenClipIt(location: Location): List<LocationPosition> {
        return this.map {
            if (planarGeometry.isPointInsidePolygon(Point2D(it.x, it.y), location.getWallPoints())) {
                return@map it
            } else {
                return@map locationPointTransformer.findPointOnLocationWallClosestToPoint(location, it)
            }
        }
    }

    private fun triangleFromTriangle2D(triangle2D: Triangle2D) =
            Triangle(triangle2D.first.toDelaunayPoint(),
                    triangle2D.second.toDelaunayPoint(),
                    triangle2D.third.toDelaunayPoint())

    private fun Location.findBeaconClosestToPoint(point: LocationPosition): LocationBeacon {
        if (beacons.isEmpty()) throw IllegalArgumentException("Cannot find closest beacon from empty list of beacons.")
        return beacons.minBy { it.position.distanceTo(point) }!!
    }

    private fun Point2D.toLocationPosition(): LocationPosition {
        return LocationPosition(this.x, this.y, 0.0)
    }

    fun LocationBeacon.toPoint(): Point2D {
        return Point2D(this.position.x, this.position.y)
    }

    fun Location.getWallPoints(): List<Point2D> {
        return this.walls.flatMap { it.toPoints() }
    }

    fun LocationWall.toPoints(): List<Point2D> {
        return listOf(Point2D(x1, y1))
    }

    fun LocationPosition.distanceTo(that: LocationPosition): Double {
        return Math.sqrt(Math.pow(this.x - that.x, 2.0) + Math.pow(this.y - that.y, 2.0))
    }

    private fun MutableList<List<Point>>.toPolygonList(): List<Polygon> {
        return this.toList().map { it.map { it.toPoint2D() } }
    }

    fun Point2D.toDelaunayPoint(): Point {
        return Point(this.x, this.y)
    }

    fun Point.toPoint2D(): Point2D {
        return Point2D(this.x(), this.y())
    }

}

