package com.estimote.cloud_plugin.api

import okhttp3.Interceptor
import okhttp3.Response
import okio.ByteString
import javax.inject.Inject

/**
 * @author Estimote Inc. (contact@estimote.com)
 */

internal class UserAgentHeaderInterceptor @Inject constructor(sdkInfo: SdkInfo,
                                                              deviceInfo: DeviceInfo) : Interceptor {
    private val userAgent =
            "${deviceInfo.appPackageName} (${deviceInfo.deviceModel}; Android ${sdkInfo.androidOsVersion}) ${sdkInfo.sdkName}/${sdkInfo.sdkVersionName} like EstimoteSDK/4.12.0"

    override fun intercept(chain: Interceptor.Chain): Response = chain.proceed(buildHeader(chain))

    private fun buildHeader(chain: Interceptor.Chain) =
            chain.request().newBuilder().addHeader("User-Agent", userAgent).build()
}

internal class ContentTypeHeaderInterceptor : Interceptor {
    override fun intercept(chain: Interceptor.Chain): Response = chain.proceed(buildHeader(chain))

    private fun buildHeader(chain: Interceptor.Chain) =
            chain.request().newBuilder().addHeader("Accept", "application/json").build()
}

internal class AuthorizationHeaderInterceptor(appId: String,
                                               appToken: String) : Interceptor {
    private val authString = ByteString.encodeUtf8("$appId:$appToken").base64()

    override fun intercept(chain: Interceptor.Chain): Response {
        return chain.proceed(buildBasicAuthHeader(chain))
    }

    private fun buildBasicAuthHeader(chain: Interceptor.Chain) =
            chain.request().newBuilder().addHeader("Authorization", "Basic $authString").build()
}

internal class CookieHeaderInterceptor(private val cookie: String) : Interceptor {

    override fun intercept(chain: Interceptor.Chain): Response {
        return chain.proceed(buildBasicAuthHeader(chain))
    }

    private fun buildBasicAuthHeader(chain: Interceptor.Chain) =
            chain.request().newBuilder().addHeader("Cookie", cookie).build()
}