package com.estimote.cloud_plugin.common.deserializer

import com.estimote.cloud_plugin.common.model.EstimoteCloudPage
import com.estimote.cloud_plugin.common.model.EstimoteCloudPageMeta
import com.google.gson.*
import java.lang.reflect.Type

/**
 * Common deserializer for Estimote Cloud endpoints with pagination support.
 * @author Estimote Inc. (contact@estimote.com)
 */
class EstimoteCloudPageDeserializer<T>(private val type: Type)
    : JsonDeserializer<EstimoteCloudPage<T>> {

    override fun deserialize(json: JsonElement?, typeOfT: Type?, context: JsonDeserializationContext?): EstimoteCloudPage<T> {
        val meta = json?.getMeta()?.let { EstimoteCloudPageMeta(it.getMetaPage(), it.getMetaTotalItems()) }
                ?: throw JsonParseException("Unable to parse paginated response: no meta value")
        val data = json?.getData()?.let { context?.deserialize<T>(it, type) }
                ?: throw JsonParseException("Unable to parse paginated response: no data value")
        return EstimoteCloudPage(meta, data)
    }

    private fun JsonElement.getData(): JsonArray? = this.asJsonObject.nullableElement("data")?.asJsonArray

    private fun JsonElement.getMeta(): JsonElement? =
            this.asJsonObject.nullableElement("meta")

    private fun JsonElement.getMetaPage(): Int =
            this.asJsonObject.nullableElement("page")?.asInt ?: 1

    private fun JsonElement.getMetaTotalItems(): Int =
            this.asJsonObject.nullableElement("total")?.asInt ?: 1

    private fun JsonObject.nullableElement(elementName: String): JsonElement? {
        val element = this.get(elementName)
        return if (element == null || element.isJsonNull) null else element
    }
}

