package com.estimote.cloud_plugin.api

import com.google.gson.Gson
import com.google.gson.GsonBuilder
import com.google.gson.JsonDeserializer
import com.google.gson.reflect.TypeToken
import com.jakewharton.retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory

/**
 * Main entry point for creating rest API objects.
 */
fun <T> createRestApi(apiClass: Class<T>,
                      baseUrl: String,
                      appId: String,
                      appToken: String,
                      sdkInfo: SdkInfo,
                      deviceInfo: DeviceInfo,
                      vararg deserializers: DeserializerConfig<*>): T {
    return Retrofit.Builder()
            .baseUrl(baseUrl)
            .addConverterFactory(GsonConverterFactory.create(createGson(*deserializers)))
            .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
            .client(getHttpClient(
                    ContentTypeHeaderInterceptor(),
                    AuthorizationHeaderInterceptor(appId, appToken),
                    UserAgentHeaderInterceptor(sdkInfo, deviceInfo)))
            .build()
            .create(apiClass)
}

private fun getHttpClient(vararg interceptors: Interceptor): OkHttpClient {
    val clientBuilder = OkHttpClient().newBuilder()
    interceptors.forEach { clientBuilder.addInterceptor(it) }
    return clientBuilder.build()
}

private fun createGson(vararg deserializers: DeserializerConfig<*>): Gson {
    return with(GsonBuilder()) {
        deserializers.forEach { registerTypeAdapter(it.typeToken.type, it.jsonDeserializer) }
        enableComplexMapKeySerialization()
        serializeNulls()
        setPrettyPrinting()
        create()
    }
}

/**
 * Data class for custom GSON deserializer objects
 */
data class DeserializerConfig<T>(val typeToken: TypeToken<T>, val jsonDeserializer: JsonDeserializer<T>)

/**
 * Defines the data model for general info about a device.
 */
data class DeviceInfo(
        val appPackageName: String,
        val deviceUniqueId: String,
        val deviceModel: String,
        val platform: String
)

/**
 * Defines basic info about an SDK
 */
data class SdkInfo(val sdkVersionName: String,
                   val sdkName: String,
                   val androidOsVersion: String)